import * as url from 'url';
import IConfig, { SubscriptionType } from '../../Display/IConfig';
import { createProgressiveWait } from '@signageos/lib/dist/Timer/progressiveWait';
import wait from '@signageos/lib/dist/Timer/wait';
import { ISignedResponse, isConfigResponseVerified } from './deviceConfigurationVerifier';
import Debug from 'debug';
import { ConfigurationChanged, ConfigurationCheckFailed, IOptionalConfig } from './deviceConfigurationActions';
import { fetch } from '../../Isomorphic/fetch';
import { put } from 'redux-saga/effects';
import { areTelemetryIntervalsSame, defaultTelemetryIntervals } from './telemetryIntervals';
import { defaultActionLimits } from './offlineActionsConfiguration';
import { defaultFeatureFlags } from './featureFlags';
const debug = Debug('@signageos/front-display:Device:Configuration:deviceConfigurationChecker');

const MIN_CHECK_INTERVAL_MS = 10e3;

export type IOptionalDynamicConfig = IOptionalConfig | (IOptionalConfig & ISignedResponse);

interface IDynamicConfig extends IOptionalConfig<string | null> {}

interface IRequiredDynamicConfig extends IConfig, IDynamicConfig {
	updatedAt: string;
	baseUrl: string;
	platformUri: string;
	socketDriver: string;
	staticBaseUrl: string;
	uploadBaseUrl: string;
	weinreUri: string;
	subscriptionType: SubscriptionType;
	checkInterval: number;
}

/**
 * @param fetchFn For tests only
 */
export function* checkDeviceConfigurationSaga(
	deviceUid: string,
	defaultConfig: IConfig,
	publicKey: string,
	fetchFn: (input: RequestInfo, init?: RequestInit) => Promise<Response> = fetch,
	threadName: 'front' | 'management',
): Generator<any, any, any> {
	let lastConfig: IDynamicConfig = {
		updatedAt: null,
		baseUrl: null,
		platformUri: null,
		socketDriver: null,
		staticBaseUrl: null,
		uploadBaseUrl: null,
		weinreUri: null,
		extendedManagementUrl: null,
		subscriptionType: null,
		checkInterval: null,
		telemetryIntervals: null,
		checkTimeBeforeConnection: false,
		offlineActionsLimits: null,
		featureFlags: null,
	};

	const errorProgressiveWait = createProgressiveWait(2 * 1e3, 2, 2 * 60e3);
	let tries: number = 0;

	while (true) {
		try {
			debug(threadName, 'Start checking', { tries });
			const config: IRequiredDynamicConfig = yield getConfig(
				deviceUid,
				defaultConfig,
				shouldSkipSsl(tries),
				publicKey,
				fetchFn,
				threadName,
			);
			debug('Retrieved config', config, lastConfig);
			errorProgressiveWait.reset(); // first retries are faster in case of fail
			const isUpdated = config.updatedAt !== lastConfig.updatedAt;
			if (
				isUpdated ||
				config.baseUrl !== lastConfig.baseUrl ||
				config.platformUri !== lastConfig.platformUri ||
				config.socketDriver !== lastConfig.socketDriver ||
				config.staticBaseUrl !== lastConfig.staticBaseUrl ||
				config.uploadBaseUrl !== lastConfig.uploadBaseUrl ||
				config.subscriptionType !== lastConfig.subscriptionType ||
				config.checkInterval !== lastConfig.checkInterval ||
				config.checkTimeBeforeConnection !== config.checkTimeBeforeConnection ||
				!areTelemetryIntervalsSame(config.telemetryIntervals, lastConfig.telemetryIntervals) ||
				config.offlineActionsLimits !== lastConfig.offlineActionsLimits ||
				config.featureFlags !== lastConfig.featureFlags
			) {
				debug(threadName, 'Config changed');
				yield put<ConfigurationChanged>({
					type: ConfigurationChanged,
					config,
				});
				lastConfig = config;
			}
			tries = 0;
			yield wait(Math.max(config.checkInterval || 0, MIN_CHECK_INTERVAL_MS));
		} catch (error) {
			debug(threadName, 'Checking erred', error);
			yield put<ConfigurationCheckFailed>({
				type: ConfigurationCheckFailed,
				error,
			});
			tries++;
			yield errorProgressiveWait.wait();
		}
	}
}

async function getConfig(
	deviceUid: string,
	defaultConfig: IConfig,
	noSsl: boolean,
	publicKey: string,
	fetchFn: (input: RequestInfo, init?: RequestInit) => Promise<Response>,
	threadName: 'front' | 'management',
): Promise<IRequiredDynamicConfig> {
	const overrideConfig = await getOverrideConfig(deviceUid, defaultConfig, noSsl, publicKey, fetchFn, threadName);
	return {
		updatedAt: overrideConfig.updatedAt,
		baseUrl: overrideConfig.baseUrl || defaultConfig.baseUrl,
		platformUri: overrideConfig.platformUri || defaultConfig.platformUri,
		socketDriver: overrideConfig.socketDriver || 'ws',
		staticBaseUrl: overrideConfig.staticBaseUrl || defaultConfig.staticBaseUrl,
		uploadBaseUrl: overrideConfig.uploadBaseUrl || defaultConfig.uploadBaseUrl,
		weinreUri: overrideConfig.weinreUri || defaultConfig.weinreUri,
		extendedManagementUrl: overrideConfig.extendedManagementUrl || defaultConfig.extendedManagementUrl,
		subscriptionType: overrideConfig.subscriptionType || defaultConfig.subscriptionType,
		checkInterval: overrideConfig.checkInterval || defaultConfig.checkInterval,
		telemetryIntervals: overrideConfig.telemetryIntervals || defaultConfig.telemetryIntervals || defaultTelemetryIntervals,
		checkTimeBeforeConnection: overrideConfig.checkTimeBeforeConnection || defaultConfig.checkTimeBeforeConnection,
		offlineActionsLimits: overrideConfig.offlineActionsLimits || defaultConfig.offlineActionsLimits || defaultActionLimits,
		featureFlags: overrideConfig.featureFlags || defaultConfig.featureFlags || defaultFeatureFlags,
	};
}

async function getOverrideConfig(
	deviceUid: string,
	defaultConfig: IConfig,
	noSsl: boolean,
	publicKey: string,
	fetchFn: (input: RequestInfo, init?: RequestInit) => Promise<Response>,
	threadName: 'front' | 'management',
): Promise<IOptionalDynamicConfig> {
	const uri = getConfigurationUri(defaultConfig.baseUrl, noSsl, deviceUid);
	try {
		debug(threadName, 'Fetching config', uri, { noSsl });
		const resp = await fetchFn(uri);
		const config: IOptionalDynamicConfig = await resp.json();
		debug(threadName, 'Fetched config', config, { noSsl });
		if (noSsl && (!('signature' in config) || !isConfigResponseVerified(config, publicKey))) {
			throw new Error('Unverified message from server');
		}
		return {
			updatedAt: config.updatedAt,
			baseUrl: null,
			platformUri: config.platformUri,
			socketDriver: config.socketDriver,
			staticBaseUrl: config.staticBaseUrl,
			uploadBaseUrl: config.uploadBaseUrl,
			weinreUri: config.weinreUri,
			extendedManagementUrl: config.extendedManagementUrl,
			subscriptionType: config.subscriptionType,
			checkInterval: config.checkInterval,
			telemetryIntervals: config.telemetryIntervals,
			checkTimeBeforeConnection: config.checkTimeBeforeConnection,
			offlineActionsLimits: config.offlineActionsLimits,
		};
	} catch (error) {
		debug(threadName, 'Erred fetching config', error);
		if (error.request && error.request.status === 404 && typeof error.body === 'object') {
			return {
				updatedAt: error.body.updatedAt,
				baseUrl: null,
				platformUri: error.body.platformUri,
				socketDriver: error.body.socketDriver,
				staticBaseUrl: error.body.staticBaseUrl,
				uploadBaseUrl: error.body.uploadBaseUrl,
				weinreUri: error.body.weirneUri,
				extendedManagementUrl: error.body.extendedManagementUrl,
				subscriptionType: error.body.subscriptionType,
				checkInterval: error.body.checkInterval,
				telemetryIntervals: error.body.telemetryIntervals,
				checkTimeBeforeConnection: error.body.checkTimeBeforeConnection,
				offlineActionsLimits: error.body.offlineActionsLimits,
				featureFlags: error.body.featureFlags,
			};
		} else {
			throw error;
		}
	}
}

function getConfigurationUri(baseUrl: string, noSsl: boolean, deviceUid: string): string {
	const configUri = `${baseUrl}/configuration/${deviceUid}`;
	if (noSsl) {
		const parsedConfigUri = url.parse(configUri, true);
		parsedConfigUri.protocol = 'http:';
		parsedConfigUri.query.signResponse = '1';
		const signedConfigUri = url.format(parsedConfigUri);

		return signedConfigUri;
	}

	return configUri;
}

function shouldSkipSsl(tries: number) {
	return (tries + 1) % 3 === 0; // Every third try is HTTP only (to try skip SSL errors)
}
