import { getCurrentTime, ITimeWithTimezone } from '../Management/Device/DateTime/dateTimeFacade';
import IManagementDriver from '../NativeDevice/Management/IManagementDriver';

export class TimeChangeDetector {
	private checkingHandler: any;
	private lastTime: ITimeWithTimezone;

	constructor(
		private managementDriver: IManagementDriver,
		private checkingIntervalMs: number,
		private significantThresholdMs: number,
	) {}

	public async start(onChange: () => void) {
		if (this.checkingHandler) {
			throw new Error(`It's running already`);
		}
		this.lastTime = await getCurrentTime(this.managementDriver);
		this.checkingHandler = setInterval(this.createCheckTimeChange(onChange), this.checkingIntervalMs);
	}

	public stop() {
		if (this.checkingHandler) {
			clearInterval(this.checkingHandler);
			this.checkingHandler = undefined;
		}
	}

	private createCheckTimeChange = (onChange: () => void) => async () => {
		const currentTime: ITimeWithTimezone = await getCurrentTime(this.managementDriver);
		const expectedCurrentTime = {
			...this.lastTime,
			timestamp: this.lastTime.timestamp + this.checkingIntervalMs,
		};
		if (this.areTimesSignificantlyDifferent(currentTime, expectedCurrentTime)) {
			onChange();
		}
		this.lastTime = currentTime;
	};

	private areTimesSignificantlyDifferent(time1: ITimeWithTimezone, time2: ITimeWithTimezone) {
		const changedByMs = Math.abs(time1.timestamp - time2.timestamp);
		return (
			changedByMs > this.significantThresholdMs ||
			time2.ntpEnabled !== time1.ntpEnabled ||
			time2.ntpServer !== time1.ntpServer ||
			time2.timezone !== time1.timezone ||
			time2.timezoneOffset !== time1.timezoneOffset
		);
	}
}
