import ICacheDriver from '../NativeDevice/ICacheDriver';
import CircularJSON from 'circular-json';

export interface IDataStorage {
	getValueOrDefault<TValue>(key: string, defaultValue: TValue): Promise<TValue>;
	setValue<TValue>(key: string, value: TValue): Promise<{ totalSizeBytes: number }>;
	removeValue(key: string): Promise<void>;
	getUsagePercentage(): Promise<number>;
}

const createDataStorage = (getNativeDriver: () => ICacheDriver): IDataStorage => ({
	async getValueOrDefault(key: string, defaultValue: any) {
		const nativeDriver = getNativeDriver();
		try {
			const rawValue = await nativeDriver.cacheGet(key);
			return JSON.parse(rawValue);
		} catch (error) {
			return defaultValue;
		}
	},
	async setValue(key: string, value: any) {
		const nativeDriver = getNativeDriver();
		const rawValue = CircularJSON.stringify(value);
		await nativeDriver.cacheSave(key, rawValue);
		return {
			// TODO is not real bytes because UTF-8 character can has more then 1 byte (Buffer is not optimized)
			totalSizeBytes: rawValue.length,
		};
	},
	async removeValue(key: string) {
		const nativeDriver = getNativeDriver();
		await nativeDriver.cacheDelete(key);
	},
	async getUsagePercentage() {
		const nativeDriver = getNativeDriver();
		const storageInfo = await nativeDriver.cacheGetStorageInfo();
		return (storageInfo.usedBytes / storageInfo.totalSizeBytes) * 100;
	},
});
export default createDataStorage;
