import { call, fork } from 'redux-saga/effects';
import Debug from 'debug';
import { whenCapable } from '../Feature/capable';
import { ManagementCapabilities } from '@signageos/common-types/dist/Device/Capabilities/ManagementCapabilities';
import ManagementCapability from '../NativeDevice/Management/ManagementCapability';
import { withDependencies } from '../DI/dependencyInjection';
import {
	ExecuteCustomScript,
	ExecuteCustomScriptSucceeded,
	ExecuteCustomScriptFailed,
} from '@signageos/actions/dist/Device/CustomScript/deviceCustomScriptActions';
import Responsibility from '../Feature/Responsibility';
import { whenResponsible } from '../Feature/responsible';
import { CustomScriptResult, isArchiveCustomScript } from '@signageos/common-types/dist/CustomScript/CustomScript';
import { CustomScriptDownloader } from './CustomScriptDownloader';
import OfflineCache from '../OfflineCache/OfflineCache';
import { IFile } from '../NativeDevice/fileSystem';
import { BrowserRuntime } from './Runtimes/browserRuntime';
import { bindAndTakeEveryOnlyWhenPlatformExperimental, putExperimental } from '../Experimental/experimentalActions';
import { getConfiguration } from './utils';
import { AppletBinaryDownloader } from '../Front/Applet/AppletBinaryDownloader';

const debug = Debug('@signageos/front-display:CustomScripts:customScriptsSaga');

// observe that the nodejs runtime is executed inside an iframe on BrightSign and Tizen
const POSSIBLE_BROWSER_RUNTIMES = ['browser', 'nodejs'];

export const customScriptsSaga = whenResponsible(
	Responsibility.CUSTOM_SCRIPTS,
	whenCapable(ManagementCapabilities.CUSTOM_SCRIPTS, customScriptsSagaDefault),
);

function* customScriptsSagaDefault(offlineCache: OfflineCache) {
	yield fork(
		whenCapable(
			ManagementCapability.CUSTOM_SCRIPTS,
			withDependencies(['managementDriver', 'staticBaseUrl'], function* ({ managementDriver, staticBaseUrl }) {
				const getStaticBaseUrl = () => {
					return staticBaseUrl;
				};

				const customScriptDownloader = new CustomScriptDownloader(offlineCache, getStaticBaseUrl);
				const appletBinaryDownloader = new AppletBinaryDownloader(false, offlineCache, getStaticBaseUrl);
				const browserRuntime = new BrowserRuntime();

				// has to be experimental for BC with the first version of the custom scripts
				yield bindAndTakeEveryOnlyWhenPlatformExperimental(ExecuteCustomScript, function* (action: ExecuteCustomScript) {
					debug('execute custom script', action);

					try {
						let customScriptFile: IFile | undefined = undefined;
						if (isArchiveCustomScript(action.customScript)) {
							customScriptFile = yield customScriptDownloader.getOrDownloadCustomScript(
								action.customScript.archiveUri,
								action.customScript.mainFile,
								action.customScript.md5Checksum,
							) ?? undefined;
						}

						let appletJsFile: IFile | undefined = undefined;
						if (action.customScript.runtime === 'browser' && action.frontAppletVersion) {
							appletJsFile = yield appletBinaryDownloader.getOrDownloadFrontAppletJsFile(action.frontAppletVersion);
						}

						debug('customScriptFile', customScriptFile);

						const configuration: Record<string, any> = yield call(
							getConfiguration,
							action.configuration,
							action.encryptedConfigurationFields || [],
							managementDriver,
						);
						debug('configuration', configuration);

						const result: CustomScriptResult = POSSIBLE_BROWSER_RUNTIMES.includes(action.customScript.runtime)
							? yield call(
									browserRuntime.runScriptInBrowser,
									managementDriver,
									action.customScript,
									configuration,
									customScriptFile,
									appletJsFile,
								)
							: yield call(
									[managementDriver.scripts, managementDriver.scripts.execute],
									action.customScript,
									configuration,
									undefined,
									customScriptFile,
								);

						debug('result', result);

						// has to be experimental for BC with the first version of the custom scripts
						yield putExperimental<ExecuteCustomScriptSucceeded>({
							type: ExecuteCustomScriptSucceeded,
							uid: action.uid,
							result,
						});
					} catch (error) {
						debug('error', error);

						// has to be experimental for BC with the first version of the custom scripts
						yield putExperimental<ExecuteCustomScriptFailed>({
							type: ExecuteCustomScriptFailed,
							uid: action.uid,
						});
					}
				});
			}),
		),
	);
}
