import { CustomScript, CustomScriptResult } from '@signageos/common-types/dist/CustomScript/CustomScript';
import { createDeferred } from '@signageos/lib/dist/Promise/deferred';
import wait from '@signageos/lib/dist/Timer/wait';

const TIMEOUT_DELAY = 90e3;

interface MessageHandler {
	messageListener: (message: MessageEvent) => void;
	getResult: () => Promise<CustomScriptResult>;
}

/**
 * ProprietaryScriptsHelper contains static methods that are used to facilitate custom scripts execution functionality in browser
 * like create, append, and remove iframe elements.
 */
export default class BrowserRuntimeHelper {
	/**
	 * Creates iframe element with absolute position placed at 0, 0 and dimensions set to 0, 0.
	 * Page content defined by script string is put into srcdoc.
	 * @param {string} script - page content specified as HTML string template
	 * @returns {HTMLIFrameElement}
	 */
	public static createIframe(script: string): HTMLIFrameElement {
		const iframe = window.document.createElement('iframe');
		iframe.style.position = 'absolute';
		iframe.style.width = '0';
		iframe.style.height = '0';
		iframe.style.top = '0';
		iframe.style.left = '0';
		iframe.srcdoc = script;
		return iframe;
	}

	/**
	 * Appends given iframe element to document's body.
	 * @param {HTMLIFrameElement} iframe
	 * @returns {void}
	 */
	public static appendIframe(iframe: HTMLIFrameElement): void {
		window.document.body.appendChild(iframe);
	}

	public static injectPlatformSpecificAPIs(
		iframe: HTMLIFrameElement,
		postResult: (result: string) => void,
		config: Record<string, any>,
		apis: Record<string, object>,
	) {
		if (!iframe.contentWindow) {
			return;
		}
		Object.defineProperty(iframe.contentWindow, 'config', { value: config });
		Object.defineProperty(iframe.contentWindow, 'postResult', { value: postResult });
		if (apis) {
			for (const [key, value] of Object.entries(apis)) {
				Object.defineProperty(iframe.contentWindow, key, { value });
			}
		}
	}

	public static addConfigToSos() {
		return `
			<script>
				Object.defineProperty(window.sos, 'config', { value: window.config });
			</script>
		`;
	}

	public static createMessageHandler(iframeId: string, postResultID: string): MessageHandler {
		let result: CustomScriptResult = {
			runtime: 'browser',
			stream: [{ pipeline: 'output', timestamp: Date.now(), data: 'OK - executed' }],
			exitCode: 0,
		};

		const waitUntilMessageReceived = createDeferred<void>(true);
		const messageListener = (message: MessageEvent) => {
			if ((message.data?.iframeId ?? '') === iframeId && (message.data?.postResultID ?? '') === postResultID) {
				result.stream[0].data = message.data.result;
				waitUntilMessageReceived.resolve();
			}
		};

		const getResult = async () => {
			await Promise.race([wait(TIMEOUT_DELAY), waitUntilMessageReceived.promise]);
			return result;
		};

		return { messageListener, getResult };
	}

	public static wrapScript(scriptUrl: string) {
		return `<script src="${scriptUrl}"></script>`;
	}

	public static checkIframeWindow = (iframe: HTMLIFrameElement): void => {
		// contentWindow is available after appending iframe.
		if (!iframe.contentWindow) {
			throw new Error('Execution failed: Cannot append iframe.');
		}
	};

	public static performCleanUp(messageListener: (message: MessageEvent) => void, iframe: HTMLIFrameElement) {
		window.document.body.removeChild(iframe);
		window.removeEventListener('message', messageListener);
	}

	public static parseError(script: CustomScript, error: Error): CustomScriptResult {
		return {
			runtime: script.runtime,
			stream: [{ pipeline: 'output', timestamp: Date.now(), data: error.message }],
			exitCode: 1,
		};
	}

	/**
	 * Returns result of custom script execution.
	 * At the time of writing, method always returns result with 'OK - executed'
	 * even though there are no guarantees that the script executed successfully.
	 * @param {CustomScript} script
	 * @returns {CustomScriptResult}
	 */
	public static getResult(script: CustomScript): CustomScriptResult {
		return {
			runtime: script.runtime,
			stream: [{ pipeline: 'output', timestamp: Date.now(), data: 'OK - executed' }],
			exitCode: 0,
		};
	}
}
