import _ from 'lodash';

const cache: { [methodName: string]: { [cacheKey: string]: any } } = {};

export function clearCache(invalidateCacheKeyPrefix: string) {
	for (const methodName in cache) {
		const cacheKeys = Object.keys(cache[methodName]);
		for (const cacheKey of cacheKeys) {
			if (cacheKey.startsWith(invalidateCacheKeyPrefix)) {
				delete cache[methodName][cacheKey];
			}
		}
	}
}

export function cachedOperation<TThis, TResult, TArgs extends any[]>(keyCallback: (self: TThis, ...args: TArgs) => string) {
	return function (_target: any, propertyKey: string, descriptor: PropertyDescriptor) {
		cache[propertyKey] = cache[propertyKey] || {};

		const originalMethod = descriptor.value;
		descriptor.value = async function (this: TThis, ...args: TArgs): Promise<TResult> {
			const cacheKey = keyCallback(this, ...args);
			if (cacheKey in cache[propertyKey]) {
				return cache[propertyKey][cacheKey];
			}
			const returnValuePromise = originalMethod.apply(this, args);
			if (!(returnValuePromise instanceof Promise)) {
				throw new Error(`cachedOperation methods must return Promise instance. But ${JSON.stringify(returnValuePromise)} given.`);
			}
			const result = await returnValuePromise;
			cache[propertyKey][cacheKey] = result;
			return result;
		};
	};
}

export function cacheInvalidating<TThis, TArgs extends any[]>(
	keysCallback: (self: TThis, ...args: TArgs) => string[],
	invalidateByKeyStart: boolean = false,
) {
	return function (_target: any, _propertyKey: string, descriptor: PropertyDescriptor) {
		const originalMethod = descriptor.value;
		descriptor.value = async function (this: TThis, ...args: TArgs) {
			const invalidateCacheKeys = keysCallback(this, ...args);
			const returnValuePromise = originalMethod.apply(this, args);
			if (!(returnValuePromise instanceof Promise)) {
				throw new Error(`cacheInvalidating methods must return Promise instance. But ${JSON.stringify(returnValuePromise)} given.`);
			}
			const result = await returnValuePromise;
			for (const methodName in cache) {
				if (invalidateByKeyStart) {
					const cacheKeys = Object.keys(cache[methodName]);
					for (const cacheKey of cacheKeys) {
						if (_.some(invalidateCacheKeys, (invalidateCacheKey: string) => cacheKey.startsWith(invalidateCacheKey))) {
							delete cache[methodName][cacheKey];
						}
					}
				} else {
					for (const invalidateCacheKey of invalidateCacheKeys) {
						delete cache[methodName][invalidateCacheKey];
					}
				}
			}
			return result;
		};
	};
}
