import ICache, { IFileContentIndex } from './ICache';

const CACHE_PREFIX: string = 'default.native_device.cache.';

export default class ProprietaryCache implements ICache {
	constructor(
		private storage: Storage | null,
		private defaultTotalSizeBytes: number,
	) {}

	public async fetchAllUids() {
		if (this.storage) {
			const uids: string[] = [];
			for (let key in this.storage) {
				if (key.indexOf(CACHE_PREFIX) === 0) {
					const uid = key.substring(CACHE_PREFIX.length);
					uids.push(uid);
				}
			}
			return uids;
		}

		console.warn(new Error('Not implemented cache get all'));
		return [];
	}

	public fetchAll() {
		return new Promise<IFileContentIndex>((resolve: (contents: IFileContentIndex) => void) => {
			if (this.storage) {
				let contents: IFileContentIndex = {};
				for (let key in this.storage) {
					if (key.indexOf(CACHE_PREFIX) === 0) {
						const value = this.storage.getItem(key)!;
						const uid = key.substring(CACHE_PREFIX.length);
						contents[uid] = value;
					}
				}
				return resolve(contents);
			} else {
				console.warn(new Error('Not implemented cache get all'));
				return resolve({});
			}
		});
	}

	public fetchOne(uid: string) {
		return new Promise<string>((resolve: (contents: string) => void, reject: (error: Error) => void) => {
			if (this.storage) {
				const content = this.storage.getItem(CACHE_PREFIX + uid);
				if (content !== null) {
					resolve(content);
				} else {
					reject(new Error('Content ' + uid + ' was not found'));
				}
			} else {
				console.warn(new Error('Not implemented cache get'));
				return reject(new Error('Content ' + uid + ' was not found'));
			}
		});
	}

	public deleteOne(uid: string) {
		if (this.storage) {
			this.storage.removeItem(CACHE_PREFIX + uid);
		} else {
			console.warn(new Error('Not implemented cache delete'));
		}
		return Promise.resolve();
	}

	public saveOne(uid: string, content: string) {
		if (this.storage) {
			if (typeof content === 'undefined') {
				throw new Error(`Store content as undefined is not allowed from uid ${uid}`);
			}
			if (content.length < 1 * 1024 * 1024) {
				this.storage.setItem(CACHE_PREFIX + uid, content);
			} else {
				console.warn('Max size for default cache saved content is 1MB');
			}
		} else {
			console.warn(new Error('Not implemented cache saveOne'));
		}
		return Promise.resolve();
	}

	public async getStorageInfo() {
		const contents = await this.fetchAll();
		const usedBytes = Object.keys(contents).reduce(
			// TODO is not real bytes because UTF-8 character can has more then 1 byte (Buffer is not optimized)
			(sum: number, key: string) => sum + key.length + (contents[key] ? contents[key].length : 0),
			0,
		);
		const totalSizeBytes = Math.max(usedBytes, this.defaultTotalSizeBytes);
		const availableBytes = totalSizeBytes - usedBytes;
		return {
			totalSizeBytes,
			availableBytes,
			usedBytes,
		};
	}
}
