import { EventEmitter } from 'events';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import { IHTMLElementProducer } from '../HTML/IHTMLElementProducer';
import IBrowser, { CloseReason, Event, EventType, IOpenLinkOptions } from '../NativeDevice/IBrowser';
import { isUrlAllowed } from './iframeHelper';

const DEBUG_NAMESPACE = '@signageos/front-display:Browser:WindowBrowser';

enum InternalEvent {
	WindowOpened = 'window_opened',
}

/**
 * Implements Browser API via opening a new browser window
 */
export default class WindowBrowser implements IBrowser, IHTMLElementProducer {
	private window: Window;
	private openedWindow: Window | null = null;
	private events: EventEmitter = new EventEmitter();
	private internalEvents: EventEmitter = new EventEmitter();

	constructor(window: Window) {
		this.window = window;
	}

	@debug(DEBUG_NAMESPACE)
	public addListener<TType extends EventType>(type: TType, cb: (event: Event<TType>) => void): void {
		this.events.addListener(type, cb);
	}

	@debug(DEBUG_NAMESPACE)
	public removeListener<TType extends EventType>(type: TType, cb: (event: Event<TType>) => void): void {
		this.events.removeListener(type, cb);
	}

	@debug(DEBUG_NAMESPACE)
	public async open(uri: string, options?: IOpenLinkOptions): Promise<void> {
		if (options?.aclMode && !isUrlAllowed(uri, options.aclMode, options.aclDomains?.map((v: string) => v.toLowerCase()) ?? [])) {
			throw new Error(`Default URL of browser ${uri} is not allowed.`);
		}
		if (this.openedWindow) {
			this.openedWindow.close();
		}

		this.openedWindow = this.window.open(uri, uri);
		if (this.openedWindow == null) {
			throw new Error(`Failed to open ${uri}`);
		}

		this.openedWindow.addEventListener('load', () => {
			this.internalEvents.emit(InternalEvent.WindowOpened, this.openedWindow);
		});
	}

	@debug(DEBUG_NAMESPACE)
	public async close(): Promise<void> {
		if (!this.openedWindow) {
			throw new Error("window doesn't exist");
		}
		try {
			this.openedWindow.close();
			this.openedWindow = null;
			this.events.emit(EventType.CLOSE, {
				type: EventType.CLOSE,
				reason: CloseReason.API,
			});
		} catch (e) {
			throw new Error("window wasn't opened by script");
		}
	}

	@debug(DEBUG_NAMESPACE)
	public onElementCreated(listener: (element: HTMLElement) => void): void {
		this.internalEvents.on(InternalEvent.WindowOpened, (window: Window) => {
			listener(window.document.body);
		});
	}

	public async isSupported(): Promise<boolean> {
		return true;
	}
}
