import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import IBrowser, { Event, EventType, IOpenLinkOptions } from '../NativeDevice/IBrowser';

const DEBUG_NAMESPACE = '@signageos/front-display:Browser:PolymorphicBrowser';

export type BrowserMethod = {
	method: string;
	browser: IBrowser;
};

/**
 * Polymorphic browser combines multiple Browser implementations and allows selection via "method" option
 *
 * This is required on some platforms that implement multiple ways to open browser with different features and limitations.
 */
export default class PolymorphicBrowser implements IBrowser {
	constructor(private browsers: BrowserMethod[]) {}

	@debug(DEBUG_NAMESPACE)
	public async open(uri: string, options?: IOpenLinkOptions): Promise<void> {
		if (this.browsers.length === 0) {
			throw new Error('no browser methods');
		}

		let index = -1;
		if (options?.method) {
			index = this.findBrowserIndexByMethod(options.method);
		}

		// if method not specified or not found, fallback to the first browser, which is default
		if (index < 0) {
			index = 0;
		}

		// to make it as smooth as possible, first open new browser and then close all others
		// only one browser can be open at a time
		await this.browsers[index].browser.open(uri, options);
		await this.closeAllBut(index);
	}

	@debug(DEBUG_NAMESPACE)
	public async close(): Promise<void> {
		for (const browser of this.browsers) {
			await browser.browser.close();
		}
	}

	public addListener<TType extends EventType>(type: TType, listener: (event: Event<TType>) => void): void {
		for (const browser of this.browsers) {
			browser.browser.addListener(type, listener);
		}
	}

	public removeListener<TType extends EventType>(type: TType, listener: (event: Event<TType>) => void): void {
		for (const browser of this.browsers) {
			browser.browser.removeListener(type, listener);
		}
	}

	public async isSupported(): Promise<boolean> {
		return this.browsers.length > 0;
	}

	private findBrowserIndexByMethod(method: string) {
		return this.browsers.findIndex((browser: BrowserMethod) => browser.method === method);
	}

	private async closeAllBut(index: number) {
		for (let i = 0; i < this.browsers.length; i++) {
			if (i !== index) {
				await this.browsers[i].browser.close();
			}
		}
	}
}
