import { bindAndTakeEvery } from '../Socket/socketActionCreator';
import { put } from 'redux-saga/effects';
import {
	AuthenticationFailed,
	SessionCreated,
	AuthenticateDevice,
	NewSessionStored,
} from '@signageos/actions/dist/Authentication/authenticationActions';
import AuthenticationType from '@signageos/actions/dist/Authentication/AuthenticationType';
import AuthenticationFailReason from '@signageos/actions/dist/Authentication/AuthenticationFailReason';
import IBasicDriver from '../NativeDevice/IBasicDriver';
import { SocketConnected } from '@signageos/actions/dist/Socket/socketActions';
import { DeviceRegistrationSucceeded, DeviceRegistrationFailed } from '@signageos/actions/dist/Device/deviceActions';
import RegisterFailReason from '@signageos/actions/dist/Device/RegisterFailReason';

export function* autoDeviceAuthentication(sessionIdKey: string, getNativeDriver: () => IBasicDriver) {
	yield bindAndTakeEvery(SocketConnected, function* () {
		try {
			const sessionId: string | null = yield getNativeDriver().getSessionId(sessionIdKey);
			if (!sessionId) {
				yield put<AuthenticationFailed>({
					type: AuthenticationFailed,
					reason: AuthenticationFailReason.MISSING_SESSION,
				});
			} else {
				yield putAuthenticateDevice(sessionId, AuthenticationType.SESSION);
			}
		} catch (error) {
			console.error('autoDeviceAuthentication failed', error);
		}
	});
}

export function* fallbackDeviceUidAuthentication(getNativeDriver: () => IBasicDriver) {
	yield bindAndTakeEvery(AuthenticationFailed, function* (action: AuthenticationFailed) {
		try {
			const deviceUid: string = yield getNativeDriver().getDeviceUid();
			if (action.reason === AuthenticationFailReason.MISSING_SESSION) {
				yield putAuthenticateDevice(deviceUid, AuthenticationType.DEVICE_UID);
			}
		} catch (error) {
			console.error('fallbackDeviceUidAuthentication failed', error);
		}
	});
}

export function* afterRegistrationDeviceUidAuthentication(getNativeDriver: () => IBasicDriver) {
	yield bindAndTakeEvery(DeviceRegistrationSucceeded, function* () {
		try {
			const deviceUid: string = yield getNativeDriver().getDeviceUid();
			yield putAuthenticateDevice(deviceUid, AuthenticationType.DEVICE_UID);
		} catch (error) {
			console.error('afterRegistrationDeviceUidAuthentication failed', error);
		}
	});
	yield bindAndTakeEvery(DeviceRegistrationFailed, function* (action: DeviceRegistrationFailed) {
		try {
			if (action.reason === RegisterFailReason.ALREADY_EXISTS) {
				const deviceUid: string = yield getNativeDriver().getDeviceUid();
				yield putAuthenticateDevice(deviceUid, AuthenticationType.DEVICE_UID);
			} else {
				console.error('Registration failed', RegisterFailReason[action.reason]);
			}
		} catch (error) {
			console.error('afterRegistrationFailedDeviceUidAuthentication failed', RegisterFailReason[action.reason]);
		}
	});
}

export function* storeDeviceSession(sessionIdKey: string, getNativeDriver: () => IBasicDriver) {
	yield bindAndTakeEvery(SessionCreated, function* (action: SessionCreated) {
		try {
			yield getNativeDriver().setSessionId(sessionIdKey, action.uid);
			yield put({ type: NewSessionStored });
		} catch (error) {
			console.error('storeDeviceSession failed', error);
		}
	});
}

function putAuthenticateDevice(identity: string, authenticationType: AuthenticationType) {
	const action: AuthenticateDevice = {
		type: AuthenticateDevice,
		credentials: { identity: identity },
		authenticationType,
	};
	// TODO delete after all servers has fallback to fixed typo in action
	(action as any).autheticationType = action.authenticationType;
	return put(action);
}
