"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.processRPC = processRPC;
exports.enqueue = enqueue;
exports.enqueueNotification = enqueueNotification;
exports.enqueueToConsumer = enqueueToConsumer;
exports.fetchNext = fetchNext;
exports.fetchAll = fetchAll;
exports.bindMore = bindMore;
exports.bindRPC = bindRPC;
exports.bindMany = bindMany;
exports.bindRejectedMany = bindRejectedMany;
exports.fetchOneRejected = fetchOneRejected;
exports.bindRejectedDetached = bindRejectedDetached;
exports.fetchOneRejectedDetached = fetchOneRejectedDetached;
exports.bindOne = bindOne;
exports.bindOneExpectingConfirmation = bindOneExpectingConfirmation;
exports.purgeOne = purgeOne;
exports.deleteMore = deleteMore;
exports.prepareMore = prepareMore;
exports.prepareMany = prepareMany;
exports.prepareRejected = prepareRejected;
exports.prepareRejectedDetached = prepareRejectedDetached;
const fetchNextMessage_1 = __importDefault(require("../fetchNextMessage"));
const ChannelProvider_1 = require("../ChannelProvider");
const EXCHANGE_PREFIX = 'events';
/** @deprecated Use @signageos/user-domain-model/dist/Lib/CQRS/commandSynchronization instead */
const EXCHANGE_NOTIFICATIONS_PREFIX = 'events_notifications';
/** Alternate exchange for events (currently unused) */
const FAILED_EXCHANGE_NAME = 'events_failed';
/** Rejected exchange for (currently bindMany only) queues */
const REJECTED_EXCHANGE_NAME = 'events_rejected';
const OPTIONS = {
    persistent: true,
    confirmable: true,
    prefetchCount: 1,
};
const createEnqueueChannel = (amqpConnection, event, exchangeName) => () => __awaiter(void 0, void 0, void 0, function* () {
    const channel = yield amqpConnection.channelProvider.getChannel(event.type, OPTIONS);
    yield channel.assertExchange(FAILED_EXCHANGE_NAME, 'direct');
    yield channel.assertExchange(exchangeName, 'fanout', { alternateExchange: FAILED_EXCHANGE_NAME });
    return channel;
});
const createBindChannel = (amqpConnection, eventType, exchangeName, options) => () => __awaiter(void 0, void 0, void 0, function* () {
    const channel = yield amqpConnection.channelProvider.getChannel(eventType, Object.assign(Object.assign({}, OPTIONS), { prefetchCount: options.prefetchCount || OPTIONS.prefetchCount }));
    yield channel.assertExchange(FAILED_EXCHANGE_NAME, 'direct');
    yield channel.assertExchange(exchangeName, 'fanout', { alternateExchange: FAILED_EXCHANGE_NAME });
    return channel;
});
const createRPCChannel = (amqpConnection, domainName, options = {}) => () => __awaiter(void 0, void 0, void 0, function* () {
    const channel = yield amqpConnection.channelProvider.getChannel(`rpc.${domainName}`, Object.assign(Object.assign({}, OPTIONS), { prefetchCount: options.prefetchCount || OPTIONS.prefetchCount }));
    return channel;
});
const createBindManyChannel = (amqpConnection, eventTypes, domainName, options) => () => __awaiter(void 0, void 0, void 0, function* () {
    const channel = yield amqpConnection.channelProvider.getChannel(domainName, Object.assign(Object.assign({}, OPTIONS), { prefetchCount: options.prefetchCount || OPTIONS.prefetchCount }));
    yield channel.assertExchange(FAILED_EXCHANGE_NAME, 'direct');
    for (const eventType of eventTypes) {
        const exchangeName = getExchangeName(options, eventType);
        yield channel.assertExchange(exchangeName, 'fanout', { alternateExchange: FAILED_EXCHANGE_NAME });
    }
    return channel;
});
const createBindRejectedChannel = (amqpConnection, options) => () => __awaiter(void 0, void 0, void 0, function* () {
    const channel = yield amqpConnection.channelProvider.getChannel(REJECTED_EXCHANGE_NAME, Object.assign(Object.assign({}, OPTIONS), { prefetchCount: options.prefetchCount || OPTIONS.prefetchCount }));
    yield channel.assertExchange(REJECTED_EXCHANGE_NAME, 'topic');
    return channel;
});
const createBindRejectedDetachedChannel = (amqpConnection, options) => () => __awaiter(void 0, void 0, void 0, function* () {
    const channel = yield amqpConnection.channelProvider.getChannel('default', Object.assign(Object.assign({}, OPTIONS), { prefetchCount: options.prefetchCount || OPTIONS.prefetchCount }));
    // currently uses default exchange
    return channel;
});
function processRPC(amqpConnection_1, domainName_1, consumerType_1, event_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, domainName, consumerType, event, messageOptions = {}, headers = {}) {
        const queueName = getRPCQueueName(consumerType, domainName);
        const response = yield amqpConnection.queuePublisher.enqueueExpectingResponse(createRPCChannel(amqpConnection, event.type), event, headers, '', // Default exchange and routingKey=queueName is used
        queueName, messageOptions);
        if (response.status === 'error') {
            throw new Error(`Event RPC processing failed: ${response.error}`);
        }
    });
}
function enqueue(amqpConnection_1, event_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, event, messageOptions = {}, headers = {}) {
        const exchangeName = getExchangeName({ notification: false }, event.type);
        yield amqpConnection.queuePublisher.enqueue(createEnqueueChannel(amqpConnection, event, exchangeName), event, headers, exchangeName, getBasicEventRoutingKey(event.type), messageOptions);
    });
}
/**
 * @deprecated Use @signageos/user-domain-model/dist/Lib/CQRS/commandSynchronization instead
 */
function enqueueNotification(amqpConnection, event) {
    return __awaiter(this, void 0, void 0, function* () {
        const exchangeName = getExchangeName({ notification: true }, event.type);
        yield amqpConnection.queuePublisher.enqueue(createEnqueueChannel(amqpConnection, event, exchangeName), event, undefined, exchangeName, getBasicEventRoutingKey(event.type), { persistent: false });
    });
}
function enqueueToConsumer(amqpConnection_1, event_1, consumerType_1, eventTypeOrDomainName_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, event, consumerType, eventTypeOrDomainName, messageOptions = {}) {
        const queueName = getQueueName(consumerType, eventTypeOrDomainName !== null && eventTypeOrDomainName !== void 0 ? eventTypeOrDomainName : event.type);
        yield amqpConnection.queuePublisher.enqueue(() => __awaiter(this, void 0, void 0, function* () {
            const channel = yield amqpConnection.channelProvider.getChannel(event.type, OPTIONS);
            return channel;
        }), event, undefined, '', // default will auto route to queue with same name as routing key
        queueName, // routingKey equal to queueName (specific consumer only)
        messageOptions);
    });
}
function fetchNext(amqpConnection_1, eventType_1, consumerType_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventType, consumerType, bindOptions = {}, queueOptions, domainName) {
        const exchangeName = getExchangeName(bindOptions, eventType);
        const queueName = getQueueName(consumerType, eventType);
        const options = Object.assign(Object.assign(Object.assign({}, queueOptions), bindOptions), (domainName ? getDomainConsumeOptions(bindOptions, consumerType, domainName) : getConsumeOptionsByBindOptions(bindOptions)));
        return yield (0, fetchNextMessage_1.default)(amqpConnection, queueName, getBasicEventRoutingKey(eventType), exchangeName, FAILED_EXCHANGE_NAME, options, 'fanout', 'direct');
    });
}
function fetchAll(amqpConnection_1, eventType_1, consumerType_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventType, consumerType, options = {}) {
        const eventsCaught = [];
        let doNext = false;
        do {
            const event = yield fetchNext(amqpConnection, eventType, consumerType, options);
            if (event) {
                doNext = true;
                eventsCaught.push(event);
            }
            else {
                doNext = false;
            }
        } while (doNext);
        return eventsCaught;
    });
}
/**
 * @deprecated Use bindMany instead.
 * This methods only calls bindOne function for every eventType.
 * If you'd like to compose consumption of more event types into one queue, see the bindMany instead.
 */
function bindMore(amqpConnection_1, eventTypes_1, consumerType_1, onEvent_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventTypes, consumerType, onEvent, options = {}) {
        for (let eventType of eventTypes) {
            yield bindOne(amqpConnection, eventType, consumerType, onEvent, options);
        }
    });
}
function bindRPC(amqpConnection_1, domainName_1, consumerType_1, onEvent_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, domainName, consumerType, onEvent, options = {}) {
        const queueName = getRPCQueueName(consumerType, domainName);
        const consumeOptions = getConsumeOptionsByBindOptions(options);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createRPCChannel(amqpConnection, domainName, options), queueName, [], // Default exchange and routingKey=queueName is used
        (event, headers) => __awaiter(this, void 0, void 0, function* () {
            try {
                yield onEvent(event, headers);
                return { status: 'success' };
            }
            catch (error) {
                return {
                    status: 'error',
                    error: error.message,
                };
            }
        }), Object.assign(Object.assign({}, consumeOptions), { respond: true }));
    });
}
/**
 * Bind more event types into one single queue.
 * This queue name contains specified domainName and is prefixed by consumerType.
 */
function bindMany(amqpConnection_1, eventTypes_1, domainName_1, consumerType_1, onEvent_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventTypes, domainName, consumerType, onEvent, options = {}) {
        const queueName = getQueueName(consumerType, domainName);
        const bindings = eventTypes.map((eventType) => {
            const exchangeName = getExchangeName(options, eventType);
            return { exchangeName, routingKey: getBasicEventRoutingKey(eventType), persistent: true };
        });
        const consumeOptions = getDomainConsumeOptions(options, consumerType, domainName);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createBindManyChannel(amqpConnection, eventTypes, domainName, options), queueName, bindings, onEvent, consumeOptions);
    });
}
/**
 * Bind all events which were rejected with enqueue=false by bindMany function (the domain type only).
 */
function bindRejectedMany(amqpConnection_1, domainName_1, consumerType_1, onEvent_1, redeliverOptions_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, domainName, consumerType, onEvent, redeliverOptions, options = {}) {
        const queueName = getRejectedManyQueueName(consumerType, domainName);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createBindRejectedChannel(amqpConnection, options), queueName, [{ exchangeName: REJECTED_EXCHANGE_NAME, routingKey: getRejectedDomainRoutingKey(consumerType, domainName), persistent: true }], onEvent, getRejectedManyConsumeOptions(options, consumerType, domainName, redeliverOptions));
    });
}
/**
 * Fetch on event which were rejected with enqueue=false by bindMany function (the domain type only).
 * Returns message containing event which has to by explicitly acked/nacked.
 */
function fetchOneRejected(amqpConnection, domainName, consumerType) {
    return __awaiter(this, void 0, void 0, function* () {
        const queueName = getRejectedManyQueueName(consumerType, domainName);
        const connection = yield amqpConnection.pool.acquire();
        const channel = yield connection.createConfirmChannel();
        const finish = () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield channel.close();
                yield amqpConnection.pool.release(connection);
            }
            catch (error) {
                yield amqpConnection.pool.destroy(connection);
                throw error;
            }
        });
        const message = yield channel.get(queueName);
        if (!message) {
            yield finish();
            return null;
        }
        const event = amqpConnection.channelProvider.decodeMessageBuffer(message.content);
        const originalQueueName = message.properties.headers['x-first-death-queue'];
        if (!originalQueueName) {
            yield finish();
            throw new Error(`Message is missing original queue name "x-first-death-queue"`);
        }
        const domain = parseQueueName(consumerType, originalQueueName);
        return {
            event,
            domain,
            ack() {
                return __awaiter(this, void 0, void 0, function* () {
                    channel.ack(message);
                    yield finish();
                });
            },
            nack(nackOptions) {
                return __awaiter(this, void 0, void 0, function* () {
                    channel.nack(message, undefined, nackOptions === null || nackOptions === void 0 ? void 0 : nackOptions.requeue);
                    yield finish();
                });
            },
        };
    });
}
/**
 * Bind one event which were rejected with enqueue=false by bindOne function (the detached type only).
 */
function bindRejectedDetached(amqpConnection_1, eventType_1, consumerType_1, onEvent_1, redeliverOptions_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventType, consumerType, onEvent, redeliverOptions, bindOptions = {}) {
        const queueName = getRejectedDetachedQueueName(consumerType, eventType);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createBindRejectedDetachedChannel(amqpConnection, bindOptions), queueName, [
            { exchangeName: '', routingKey: queueName, persistent: true }, // this is default binding defined by AMQP protocol
        ], onEvent, getRejectedDetachedConsumeOptions(bindOptions, consumerType, eventType, redeliverOptions));
    });
}
/**
 * Fetch on event which were rejected with enqueue=false by bindOne function (the detached type only).
 * Returns message containing event which has to by explicitly acked/nacked.
 */
function fetchOneRejectedDetached(amqpConnection, eventType, consumerType) {
    return __awaiter(this, void 0, void 0, function* () {
        const queueName = getRejectedDetachedQueueName(consumerType, eventType);
        const connection = yield amqpConnection.pool.acquire();
        const channel = yield connection.createConfirmChannel();
        const finish = () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield channel.close();
                yield amqpConnection.pool.release(connection);
            }
            catch (error) {
                yield amqpConnection.pool.destroy(connection);
                throw error;
            }
        });
        const message = yield channel.get(queueName);
        if (!message) {
            yield finish();
            return null;
        }
        const event = amqpConnection.channelProvider.decodeMessageBuffer(message.content);
        return {
            event,
            ack() {
                return __awaiter(this, void 0, void 0, function* () {
                    channel.ack(message);
                    yield finish();
                });
            },
            nack(nackOptions) {
                return __awaiter(this, void 0, void 0, function* () {
                    channel.nack(message, undefined, nackOptions === null || nackOptions === void 0 ? void 0 : nackOptions.requeue);
                    yield finish();
                });
            },
        };
    });
}
function bindOne(amqpConnection_1, eventType_1, consumerType_1, onEvent_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventType, consumerType, onEvent, options = {}) {
        const exchangeName = getExchangeName(options, eventType);
        const queueName = getQueueName(consumerType, eventType);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createBindChannel(amqpConnection, eventType, exchangeName, options), queueName, [{ exchangeName, routingKey: getBasicEventRoutingKey(eventType), persistent: true }], onEvent, getConsumeOptionsByBindOptions(options));
    });
}
function bindOneExpectingConfirmation(amqpConnection_1, eventType_1, consumerType_1, onEvent_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventType, consumerType, onEvent, options = {}) {
        const exchangeName = getExchangeName(options, eventType);
        const queueName = getQueueName(consumerType, eventType);
        return yield amqpConnection.queueSubscriber.subscribeExpectingConfirmationRepeatable(createBindChannel(amqpConnection, eventType, exchangeName, options), queueName, [{ exchangeName, routingKey: getBasicEventRoutingKey(eventType), persistent: true }], (event, _headers, ack, nack) => __awaiter(this, void 0, void 0, function* () {
            const ackAsync = () => __awaiter(this, void 0, void 0, function* () { return ack(); });
            yield onEvent(event, ackAsync, nack);
        }), getConsumeOptionsByBindOptions(options));
    });
}
function purgeOne(amqpConnection, eventType, consumerType) {
    return __awaiter(this, void 0, void 0, function* () {
        const queueName = getQueueName(consumerType, eventType);
        const channel = yield amqpConnection.channelProvider.getChannel(eventType);
        try {
            yield channel.purge(queueName);
        }
        finally {
            yield channel.close();
        }
    });
}
function deleteMore(amqpConnection, eventTypes, consumerType) {
    return __awaiter(this, void 0, void 0, function* () {
        for (let eventType of eventTypes) {
            const queueName = getQueueName(consumerType, eventType);
            const channel = yield amqpConnection.channelProvider.getChannel(eventType);
            try {
                yield channel.delete(queueName);
            }
            finally {
                yield channel.close();
            }
        }
    });
}
function prepareMore(amqpConnection_1, events_1, consumerType_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, events, consumerType, options = {}) {
        for (const event of events) {
            // Hack to create event queue for exchange
            yield fetchNext(amqpConnection, event, consumerType, options);
        }
    });
}
function prepareMany(amqpConnection_1, events_1, domainName_1, consumerType_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, events, domainName, consumerType, options = {}) {
        const cancel = yield bindMany(amqpConnection, events, domainName, consumerType, () => __awaiter(this, void 0, void 0, function* () { return new Promise(() => undefined); }), options);
        const close = yield cancel();
        yield close();
    });
}
function prepareRejected(amqpConnection_1, domainName_1, consumerType_1, redeliverOptions_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, domainName, consumerType, redeliverOptions, options = {}) {
        const cancel = yield bindRejectedMany(amqpConnection, domainName, consumerType, () => __awaiter(this, void 0, void 0, function* () { return new Promise(() => undefined); }), redeliverOptions, options);
        const close = yield cancel();
        yield close();
    });
}
function prepareRejectedDetached(amqpConnection_1, eventType_1, consumerType_1, redeliverOptions_1) {
    return __awaiter(this, arguments, void 0, function* (amqpConnection, eventType, consumerType, redeliverOptions, bindOptions = {}) {
        const cancel = yield bindRejectedDetached(amqpConnection, eventType, consumerType, () => __awaiter(this, void 0, void 0, function* () { return new Promise(() => undefined); }), redeliverOptions, bindOptions);
        const close = yield cancel();
        yield close();
    });
}
function getDomainConsumeOptions(options, consumerType, domainName) {
    return Object.assign(Object.assign({}, getConsumeOptionsByBindOptions(options)), { deadLetterExchange: REJECTED_EXCHANGE_NAME, deadLetterRoutingKey: getRejectedDomainRoutingKey(consumerType, domainName) });
}
function getRejectedManyConsumeOptions(bindOptions, consumerType, domainName, redeliverOptions) {
    const queueName = getQueueName(consumerType, domainName);
    return Object.assign(Object.assign({}, getConsumeOptionsByBindOptions(bindOptions)), { deadLetterExchange: '', deadLetterRoutingKey: queueName, messageExpiresMs: redeliverOptions.redeliverDelayMs, queueType: 'classic' });
}
function getRejectedDetachedConsumeOptions(bindOptions, consumerType, eventType, redeliverOptions) {
    const queueName = getQueueName(consumerType, eventType);
    return Object.assign(Object.assign({}, getConsumeOptionsByBindOptions(bindOptions)), { deadLetterExchange: '', deadLetterRoutingKey: queueName, messageExpiresMs: redeliverOptions.redeliverDelayMs, queueType: 'classic' });
}
function getConsumeOptionsByBindOptions(options) {
    return {
        autoDelete: typeof options.persistent !== 'undefined' ? !options.persistent : false,
        exclusive: typeof options.exclusive !== 'undefined' ? options.exclusive : false,
        redeliverDelayMs: typeof options.redeliverDelayMs !== 'undefined' ? options.redeliverDelayMs : 1e3,
        suppressFirstError: typeof options.suppressFirstError !== 'undefined' ? options.suppressFirstError : true,
        singleActiveConsumer: options.singleActiveConsumer,
        deadLetterIfRedeliveredAndErred: options.deadLetterIfRedeliveredAndErred,
        deadLetterIfErred: options.deadLetterIfErred,
        queueType: typeof options.queueType !== 'undefined' ? options.queueType : 'quorum', // default for event queues are swapped
    };
}
function getExchangeName(options, eventType) {
    const prefix = options.notification ? EXCHANGE_NOTIFICATIONS_PREFIX : EXCHANGE_PREFIX;
    return `${prefix}-${eventType}`;
}
function getQueueName(consumerType, eventTypeOrDomainName) {
    return consumerType + '_' + eventTypeOrDomainName;
}
function getRPCQueueName(consumerType, domainName) {
    return consumerType + '_' + domainName + '.rpc';
}
function parseQueueName(consumerType, queueName) {
    const eventTypeOrDomainName = queueName.substring(consumerType.length + 1);
    const expectedQueueName = getQueueName(consumerType, eventTypeOrDomainName);
    if (expectedQueueName !== queueName) {
        throw new Error(`Incorrect queue name cannot be parsed: ${consumerType}, ${queueName}`);
    }
    return eventTypeOrDomainName;
}
function getRejectedManyQueueName(consumerType, domainName) {
    return 'events_rejected.' + getQueueName(consumerType, domainName);
}
function getRejectedDetachedQueueName(consumerType, eventType) {
    return ChannelProvider_1.REJECTED_QUEUE_PREFIX + getQueueName(consumerType, eventType);
}
function escapeEventTypeForRoutingKey(eventType) {
    return eventType.replace(/\./g, '_');
}
function getBasicEventRoutingKey(eventType) {
    return 'event.' + escapeEventTypeForRoutingKey(eventType);
}
function getRejectedDomainRoutingKey(consumerType, domainName) {
    return 'domain.' + consumerType + '.' + domainName;
}
//# sourceMappingURL=eventQueue.js.map