"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomScriptResultSchema = exports.ResultStreamPipelineSchema = exports.CustomScriptSchema = exports.RuntimeSchema = exports.DangerLevelSchema = exports.SCRIPT_MAX_LENGTH = void 0;
exports.isInlineCustomScript = isInlineCustomScript;
exports.isArchiveCustomScript = isArchiveCustomScript;
const zod_1 = require("zod");
exports.SCRIPT_MAX_LENGTH = 10e3;
exports.DangerLevelSchema = zod_1.z.enum(['low', 'medium', 'high', 'critical']);
exports.RuntimeSchema = zod_1.z.enum([
    /** PowerShell */
    'ps1',
    /** Bash */
    'bash',
    /** Shell */
    'sh',
    /** Node.js */
    'nodejs',
    /** Browser JS */
    'browser',
    /** BrightScript */
    'brs',
]);
const InlineCustomScriptSchema = zod_1.z.strictObject({
    runtime: exports.RuntimeSchema,
    /** The script content */
    script: zod_1.z.string().max(exports.SCRIPT_MAX_LENGTH),
});
const ArchiveCustomScriptSchema = zod_1.z.strictObject({
    runtime: exports.RuntimeSchema,
    /** URI to download code archive */
    archiveUri: zod_1.z.string().url(),
    /** MD5 checksum of the archive */
    md5Checksum: zod_1.z.string(),
    /** The entry point of the script */
    mainFile: zod_1.z.string(),
});
exports.CustomScriptSchema = zod_1.z.union([InlineCustomScriptSchema, ArchiveCustomScriptSchema]);
function isInlineCustomScript(script) {
    return 'script' in script;
}
function isArchiveCustomScript(script) {
    return 'archiveUri' in script;
}
/**
 * @see https://learn.microsoft.com/en-us/powershell/module/microsoft.powershell.utility/write-output?view=powershell-7.4
 */
exports.ResultStreamPipelineSchema = zod_1.z.enum(['error', 'output']);
exports.CustomScriptResultSchema = zod_1.z.strictObject({
    runtime: exports.RuntimeSchema,
    /**
     * The output/error stream of the script ordered chunks by the timestamp.
     */
    stream: zod_1.z.array(zod_1.z.object({
        /**
         * The pipeline type of the output/error stream.
         */
        pipeline: exports.ResultStreamPipelineSchema,
        /**
         * The exact time in milliseconds when the chunk of output/error stream was written to the console.
         * It is required to properly order the output/error stream on the target UI.
         */
        timestamp: zod_1.z.number().int(),
        /**
         * The data chunk of the output/error stream.
         * Usually, the chunk can be single line of the output/error stream.
         */
        data: zod_1.z.string(),
    })),
    /**
     * The exit code of the script when finished.
     * Usually, 0 means success, and any other value means an error.
     */
    exitCode: zod_1.z.number().int(),
});
