"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const debug_1 = __importDefault(require("debug"));
const debugDecorator_1 = require("@signageos/lib/dist/Debug/debugDecorator");
const lodash_1 = require("lodash");
const DEBUG_NAMESPACE = '@signageos/front-display:Video:HTMLVideo';
const logDebug = (0, debug_1.default)(DEBUG_NAMESPACE);
const STREAM_RECONNECT_TIMEOUT = 2e3;
class HTMLVideo {
    constructor(window, foregroundWrapperElement = window.document.body, backgroundWrapperElement) {
        this.window = window;
        this.foregroundWrapperElement = foregroundWrapperElement;
        this.backgroundWrapperElement = backgroundWrapperElement;
        this.endStreamWhenCannotReconnectDebounce = (0, lodash_1.debounce)(() => this.eventEmitter.emit('ended'), STREAM_RECONNECT_TIMEOUT);
        this.endIfCannotRecover = () => {
            this.video.addEventListener('stalled', this.endStreamWhenCannotReconnectDebounce);
            this.video.addEventListener('error', this.endStreamWhenCannotReconnectDebounce);
            this.video.addEventListener('playing', this.endStreamWhenCannotReconnectDebounce.cancel);
        };
        this.eventEmitter = new events_1.EventEmitter();
        this.initializeVideo();
        this.resetEventEmitter();
    }
    play() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield this.playVideo();
            }
            catch (error) {
                logDebug('playVideo error', error);
                this.ifChromeLogWarningWithLinkToDocs();
                yield this.playVideoMutedFallback();
            }
            this.show();
        });
    }
    stop() {
        return __awaiter(this, void 0, void 0, function* () {
            // https://html.spec.whatwg.org/multipage/media.html#best-practices-for-authors-using-media-elements
            this.clearVideoTagStyle();
            this.video.pause();
            this.video.removeAttribute('src');
            this.video.load();
            this.eventEmitter.emit('stopped');
        });
    }
    pause() {
        return __awaiter(this, void 0, void 0, function* () {
            this.video.pause();
        });
    }
    resume() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.video.play();
        });
    }
    prepare(uri, x, y, width, height, options = {}) {
        this.currentOptions = options;
        return new Promise((resolve, reject) => {
            this.setupVideoObjectStyle(x, y, width, height);
            if (typeof options.volume !== 'undefined') {
                this.video.volume = options.volume / 100;
            }
            else {
                this.video.volume = 1;
            }
            this.prepareSequenceListeners()
                .then(() => {
                this.endIfCannotRecover();
                resolve();
            })
                .catch((e) => reject(e));
            // Wait one tick because prepareSequenceListeners is in new Promise which delays by 1 tick as well
            setTimeout(() => this.setVideoSourceAndLoad(uri));
        });
    }
    addEventListener(eventName, listener) {
        this.eventEmitter.addListener(eventName, listener);
    }
    removeAllEventListeners() {
        this.resetEventEmitter();
    }
    setVolume(volumePercentage) {
        this.video.volume = volumePercentage / 100;
    }
    getDuration() {
        return this.video.duration * 1e3;
    }
    playVideo() {
        return __awaiter(this, void 0, void 0, function* () {
            this.video.muted = false;
            this.video.currentTime = 0;
            yield this.video.play();
        });
    }
    playVideoMutedFallback() {
        return __awaiter(this, void 0, void 0, function* () {
            this.video.muted = true;
            this.video.currentTime = 0;
            yield this.video.play();
        });
    }
    /**
     * This method loads video wrapped by this HTML video class.
     *
     * Set src and load has to be called in one tick otherwise our listeners logic is broken
     * by browser auto load behavior which we can not change.
     *
     * More on https://developer.mozilla.org/en-US/docs/Web/API/HTMLMediaElement/loadstart_event
     * and here https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-preload.
     */
    setVideoSourceAndLoad(uri) {
        this.video.setAttribute('src', uri);
        this.video.load();
    }
    hide() {
        this.videoWrapper.style.visibility = 'hidden';
        if (this.videoWrapper.parentElement !== this.foregroundWrapperElement) {
            this.foregroundWrapperElement.appendChild(this.videoWrapper);
        }
    }
    show() {
        if (this.currentOptions && this.currentOptions.background && this.backgroundWrapperElement) {
            this.backgroundWrapperElement.appendChild(this.videoWrapper);
        }
        this.videoWrapper.style.visibility = 'visible';
    }
    clearVideoTagStyle() {
        this.videoWrapper.style.left = '0px';
        this.videoWrapper.style.top = '0px';
        this.videoWrapper.style.width = '0px';
        this.videoWrapper.style.height = '0px';
        this.hide();
    }
    setupVideoObjectStyle(x, y, width, height) {
        this.videoWrapper.setAttribute('width', width.toString());
        this.videoWrapper.setAttribute('height', height.toString());
        this.videoWrapper.style.left = x + 'px';
        this.videoWrapper.style.top = y + 'px';
        this.videoWrapper.style.width = width + 'px';
        this.videoWrapper.style.height = height + 'px';
    }
    initializeVideo() {
        this.videoWrapper = this.window.document.createElement('div');
        this.videoWrapper.style.background = '#000000';
        this.videoWrapper.style.position = 'absolute';
        this.hide();
        this.foregroundWrapperElement.appendChild(this.videoWrapper);
        this.video = this.window.document.createElement('video');
        this.video.setAttribute('class', 'default-video');
        this.video.setAttribute('width', '100%');
        this.video.setAttribute('height', '100%');
        this.video.style.width = '100%';
        this.videoWrapper.appendChild(this.video);
        this.video.addEventListener('ended', () => {
            this.eventEmitter.emit('ended');
        });
        this.video.addEventListener('error', () => {
            this.eventEmitter.emit('error');
        });
        this.debugVideo();
    }
    resetEventEmitter() {
        this.eventEmitter.removeAllListeners();
        // "error" event type is treated as a special case and has to have at least one listener or it can crash the whole process
        // https://nodejs.org/api/events.html#events_error_events
        this.eventEmitter.addListener('error', () => {
            /* do nothing */
        });
    }
    ifChromeLogWarningWithLinkToDocs() {
        const isChrome = !!this.window.chrome;
        if (isChrome) {
            console.warn('It looks like an error occurred during video playback. ' +
                'View some of the common causes here https://docs.signageos.io/hc/en-us/articles/4405238997138.');
        }
    }
    prepareSequenceListeners() {
        return new Promise((resolve, reject) => {
            let canPlayThroughListener, emptiedListener, preloadEnded;
            canPlayThroughListener = () => {
                this.video.removeEventListener('canplaythrough', canPlayThroughListener);
                this.video.removeEventListener('emptied', emptiedListener);
                this.video.removeEventListener('stalled', preloadEnded);
                resolve();
            };
            emptiedListener = () => {
                this.video.removeEventListener('emptied', emptiedListener);
                this.video.removeEventListener('canplaythrough', canPlayThroughListener);
                this.video.removeEventListener('stalled', preloadEnded);
                reject(new Error('Video status changed before it could finish prepare'));
            };
            const loadstartListener = () => {
                this.video.removeEventListener('loadstart', loadstartListener);
                this.video.addEventListener('canplaythrough', canPlayThroughListener);
                this.video.addEventListener('emptied', emptiedListener);
            };
            preloadEnded = () => {
                this.video.removeEventListener('loadstart', loadstartListener);
                this.video.removeEventListener('stalled', preloadEnded);
                this.video.removeEventListener('emptied', emptiedListener);
                this.video.removeEventListener('canplaythrough', canPlayThroughListener);
                reject(new Error("Couldn't connect to video"));
            };
            this.video.addEventListener('loadstart', loadstartListener);
            this.video.addEventListener('stalled', preloadEnded);
        });
    }
    debugVideo() {
        const events = [
            'canplay',
            'canplaythrough',
            'durationchange',
            'emptied',
            'ended',
            'error',
            'loadeddata',
            'loadedmetadata',
            'loadstart',
            'pause',
            'play',
            'playing',
            'progress',
            'ratechange',
            'seeked',
            'seeking',
            'stalled',
            'suspend',
            'timeupdate',
            'volumechange',
            'waiting',
            'complete',
        ];
        for (const event of events) {
            this.video.addEventListener(event, (...args) => {
                logDebug(`Video event: ${event}`, ...args);
            });
        }
    }
}
exports.default = HTMLVideo;
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], HTMLVideo.prototype, "play", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], HTMLVideo.prototype, "stop", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], HTMLVideo.prototype, "pause", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], HTMLVideo.prototype, "resume", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String, Number, Number, Number, Number, Object]),
    __metadata("design:returntype", void 0)
], HTMLVideo.prototype, "prepare", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Number]),
    __metadata("design:returntype", void 0)
], HTMLVideo.prototype, "setVolume", null);
//# sourceMappingURL=HTMLVideo.js.map