"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createEmptyTestResult = exports.runTests = exports.afterEach = exports.beforeEach = exports.after = exports.before = exports.skip = exports.it = exports.describe = exports.Describe = void 0;
const timeout_1 = require("@signageos/lib/dist/Timer/timeout");
const debug_1 = __importDefault(require("debug"));
const debug = (0, debug_1.default)('@signageos/front-display:Test:TestFramework');
const DEFAULT_TIMEOUT = 120e3;
var TestItemType;
(function (TestItemType) {
    TestItemType["TITLE"] = "Title";
    TestItemType["DESCRIBE"] = "Describe";
    TestItemType["TEST"] = "Test";
    TestItemType["BEFORE"] = "Before";
    TestItemType["BEFORE_EACH"] = "BeforeEach";
    TestItemType["AFTER"] = "After";
    TestItemType["AFTER_EACH"] = "AfterEach";
})(TestItemType || (TestItemType = {}));
class SkipError extends Error {
    constructor(reason) {
        super(reason);
        // https://github.com/Microsoft/TypeScript/wiki/Breaking-Changes#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, SkipError.prototype);
    }
}
class Describe {
    constructor(title, run, timeout) {
        this.title = title;
        this.run = run;
        this.timeout = timeout;
        this.type = TestItemType.DESCRIBE;
    }
}
exports.Describe = Describe;
function describe(title, describeCallback, timeout) {
    return new Describe(title, describeCallback, timeout);
}
exports.describe = describe;
class It {
    constructor(title, run, timeout) {
        this.title = title;
        this.run = run;
        this.timeout = timeout;
        this.type = TestItemType.TEST;
    }
}
function it(title, describeCallback, timeout) {
    return new It(title, describeCallback, timeout);
}
exports.it = it;
function skip(reason) {
    throw new SkipError(reason);
}
exports.skip = skip;
class Before {
    constructor(title, run, timeout) {
        this.title = title;
        this.run = run;
        this.timeout = timeout;
        this.type = TestItemType.BEFORE;
    }
}
function before(title, describeCallback, timeout) {
    return new Before(title, describeCallback, timeout);
}
exports.before = before;
class After {
    constructor(title, run, timeout) {
        this.title = title;
        this.run = run;
        this.timeout = timeout;
        this.type = TestItemType.AFTER;
    }
}
function after(title, describeCallback, timeout) {
    return new After(title, describeCallback, timeout);
}
exports.after = after;
class BeforeEach {
    constructor(title, run, timeout) {
        this.title = title;
        this.run = run;
        this.timeout = timeout;
        this.type = TestItemType.BEFORE_EACH;
    }
}
function beforeEach(title, describeCallback, timeout) {
    return new BeforeEach(title, describeCallback, timeout);
}
exports.beforeEach = beforeEach;
class AfterEach {
    constructor(title, run, timeout) {
        this.title = title;
        this.run = run;
        this.timeout = timeout;
        this.type = TestItemType.AFTER_EACH;
    }
}
function afterEach(title, describeCallback, timeout) {
    return new AfterEach(title, describeCallback, timeout);
}
exports.afterEach = afterEach;
function loadTestSuite(testPayload) {
    const creatingTestSuite = { tests: [], describes: [], before: [], after: [], beforeEach: [], afterEach: [] };
    for (let item of testPayload) {
        switch (item.type) {
            case TestItemType.TEST:
                creatingTestSuite.tests.push(item);
                break;
            case TestItemType.DESCRIBE:
                creatingTestSuite.describes.push(item);
                break;
            case TestItemType.BEFORE:
                creatingTestSuite.before.push(item);
                break;
            case TestItemType.AFTER:
                creatingTestSuite.after.push(item);
                break;
            case TestItemType.BEFORE_EACH:
                creatingTestSuite.beforeEach.push(item);
                break;
            case TestItemType.AFTER_EACH:
                creatingTestSuite.afterEach.push(item);
                break;
            default:
                throw new Error(`Unknown TestItemType: ${item.type}`);
        }
    }
    return creatingTestSuite;
}
function runTests(suite) {
    return __awaiter(this, void 0, void 0, function* () {
        const testSuite = loadTestSuite(suite.run());
        const result = createEmptyTestResult(suite.title);
        // describe - BEFORE
        // eslint-disable-next-line @typescript-eslint/no-shadow
        const { before, failed, skipped } = yield runBefore(testSuite.before);
        result.before = before;
        if (failed || skipped) {
            result.total++;
            if (failed) {
                result.failed++;
            }
            if (skipped) {
                result.skipped++;
            }
            return result;
        }
        // describe - nested Describe
        for (let describe of testSuite.describes) {
            const describeResult = yield runTests(describe);
            result.describe.push(describeResult);
            result.total += describeResult.total;
            result.successful += describeResult.successful;
            result.skipped += describeResult.skipped;
            result.failed += describeResult.failed;
            result.duration += describeResult.duration;
        }
        // describe - tests
        const tests = yield runActualTests(testSuite);
        result.test = tests;
        for (let test of tests) {
            result.total++;
            result.duration += test.test.duration;
            let beforeEachFailed = false;
            let beforeEachSkipped = false;
            for (let beforeEach of test.beforeEach) {
                if (beforeEach.failed) {
                    beforeEachFailed = true;
                }
                if (beforeEach.skipped) {
                    beforeEachSkipped = true;
                }
            }
            if (beforeEachFailed) {
                result.failed++;
                continue;
            }
            if (beforeEachSkipped) {
                result.skipped++;
                continue;
            }
            if (test.test.failed) {
                result.failed++;
                continue;
            }
            if (test.test.skipped) {
                result.skipped++;
                continue;
            }
            let afterEachFailed = false;
            let afterEachSkipped = false;
            for (let afterEach of test.afterEach) {
                if (afterEach.failed) {
                    afterEachFailed = true;
                }
                if (afterEach.skipped) {
                    afterEachSkipped = true;
                }
            }
            if (afterEachFailed) {
                result.failed++;
                continue;
            }
            if (afterEachSkipped) {
                result.skipped++;
                continue;
            }
            // consider test being successful - neither skip nor failure
            result.successful++;
        }
        // describe - after - cleaning - failures/skips in here do not modify the overall test results
        result.after = yield runItems(testSuite.after);
        return result;
    });
}
exports.runTests = runTests;
function createEmptyTestResult(title) {
    return {
        title,
        total: 0,
        successful: 0,
        failed: 0,
        skipped: 0,
        test: [],
        describe: [],
        before: [],
        after: [],
        duration: 0,
    };
}
exports.createEmptyTestResult = createEmptyTestResult;
function runActualTests(testSuite) {
    return __awaiter(this, void 0, void 0, function* () {
        const testResults = [];
        for (let test of testSuite.tests) {
            const testResult = { title: test.title, skipped: false, failed: false, duration: 0 };
            let shouldPreventRun = false;
            const beforeEachResults = yield runItems(testSuite.beforeEach);
            for (const beforeEach of beforeEachResults) {
                if (beforeEach.failed) {
                    testResult.failed = true;
                    testResult.reason = `Before each error. Reason: ${beforeEach.reason}`;
                    shouldPreventRun = true;
                }
                if (beforeEach.skipped) {
                    testResult.skipped = true;
                    testResult.reason = `Before each skipped. Reason: ${beforeEach.reason}`;
                    shouldPreventRun = true;
                }
            }
            if (shouldPreventRun) {
                testResults.push({ beforeEach: beforeEachResults, test: testResult, afterEach: [] });
                continue;
            }
            const timeStart = new Date().getTime();
            try {
                yield (0, timeout_1.withTimeout)(test.run(), test.timeout || DEFAULT_TIMEOUT);
            }
            catch (error) {
                testResult.reason = error.message;
                if (error instanceof SkipError) {
                    testResult.skipped = true;
                }
                else {
                    testResult.failed = true;
                }
            }
            testResult.duration = new Date().getTime() - timeStart;
            const afterEachResult = yield runItems(testSuite.afterEach);
            testResults.push({ beforeEach: beforeEachResults, test: testResult, afterEach: afterEachResult });
        }
        return testResults;
    });
}
function runBefore(items) {
    return __awaiter(this, void 0, void 0, function* () {
        const itemResults = [];
        for (let item of items) {
            const result = yield executeItem(item);
            itemResults.push(result);
            if (result.skipped) {
                // skip all following on skip
                return { before: itemResults, failed: false, skipped: true };
            }
            if (result.failed) {
                // skip all following on failure
                return { before: itemResults, failed: true, skipped: true };
            }
        }
        return { before: itemResults, failed: false, skipped: false };
    });
}
function runItems(items) {
    return __awaiter(this, void 0, void 0, function* () {
        const itemResults = [];
        for (let item of items) {
            itemResults.push(yield executeItem(item));
        }
        return itemResults;
    });
}
function executeItem(instance) {
    return __awaiter(this, void 0, void 0, function* () {
        const result = { title: instance.title, skipped: false, failed: false, duration: 0 };
        try {
            debug('run test: ' + instance.title);
            yield (0, timeout_1.withTimeout)(instance.run(), instance.timeout || DEFAULT_TIMEOUT);
            debug('test successful: ' + instance.title);
            return result;
        }
        catch (error) {
            result.reason = error.message;
            if (error instanceof SkipError) {
                debug('test skipped: ' + instance.title);
                result.skipped = true;
                return result;
            }
            debug(`test failed: ${instance.title}, reason: ${error.message}`);
            result.failed = true;
            return result;
        }
    });
}
//# sourceMappingURL=TestFramework.js.map