"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Group = void 0;
const lockedDecorator_1 = require("@signageos/lib/es6/Lock/lockedDecorator");
const debug_1 = __importDefault(require("debug"));
const events_1 = require("events");
const lodash_1 = require("lodash");
const IPeerNetwork_1 = require("../../PeerNetwork/IPeerNetwork");
const IGroup_1 = require("./IGroup");
const debug = (0, debug_1.default)('@signageos/front-display:Synchronization:Group');
var MessageType;
(function (MessageType) {
    MessageType["Query"] = "query";
    MessageType["JoinGroup"] = "join_group";
    MessageType["LeaveGroup"] = "leave_group";
    // eslint-disable-next-line @typescript-eslint/no-shadow
    MessageType["GroupData"] = "group_data";
})(MessageType || (MessageType = {}));
function isMessage(message) {
    return typeof message === 'object' && message !== null && 'type' in message;
}
function isQueryMessage(message) {
    return isMessage(message) && message.type === MessageType.Query;
}
function isJoinGroupMessage(message) {
    return isMessage(message) && message.type === MessageType.JoinGroup;
}
function isLeaveGroupMessage(message) {
    return isMessage(message) && message.type === MessageType.LeaveGroup;
}
function isGroupDataMessage(message) {
    return isMessage(message) && message.type === MessageType.GroupData;
}
/**
 * Group automatically handles membership in a group for synchronization purposes.
 */
class Group {
    constructor({ name, groupName, peerNetwork, sendQueryIntervalMs, }) {
        this.peers = [];
        this.started = false;
        this.joinedAt = new Date();
        this.emitter = new events_1.EventEmitter();
        this.sendQueryInterval = null;
        this.memberName = name;
        this.groupName = groupName;
        this.peerNetwork = peerNetwork;
        this.handleMessage = this.handleMessage.bind(this);
        this.handlePeersChanged = this.handlePeersChanged.bind(this);
        this.sendQueryIntervalMs = sendQueryIntervalMs || 30e3;
    }
    getGroupName() {
        return this.groupName;
    }
    getMe() {
        const me = this.peerNetwork.getMe();
        return {
            id: me.id,
            name: this.memberName,
            joinedAt: this.joinedAt,
        };
    }
    getPeers() {
        return this.peers;
    }
    getAllMembers() {
        const me = this.getMe();
        const peers = this.getPeers();
        return [me, ...peers];
    }
    getMaster() {
        const allMembers = this.getAllMembers();
        // sort members by joinedAt
        const membersSorted = allMembers.sort((a, b) => a.joinedAt.getTime() - b.joinedAt.getTime());
        return membersSorted[0];
    }
    join() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.started) {
                throw new Error('group already joined');
            }
            this.peerNetwork.addListener(IPeerNetwork_1.PeerNetworkEvent.Message, this.handleMessage);
            this.peerNetwork.addListener(IPeerNetwork_1.PeerNetworkEvent.PeersChanged, this.handlePeersChanged);
            this.joinedAt = new Date();
            try {
                yield this.sendQueryMessage();
                yield this.sendJoinGroupMessage();
            }
            catch (error) {
                // we need to start listening to the events before sending the messages to prevent race conditions
                // but if something fails, we need to cleanup
                this.peerNetwork.removeListener(IPeerNetwork_1.PeerNetworkEvent.Message, this.handleMessage);
                this.peerNetwork.removeListener(IPeerNetwork_1.PeerNetworkEvent.PeersChanged, this.handlePeersChanged);
                throw error;
            }
            this.startSendingQueryMessages();
            this.started = true;
        });
    }
    leave() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.started) {
                throw new Error('group not joined');
            }
            this.stopSendingQueryMessages();
            this.peerNetwork.removeListener(IPeerNetwork_1.PeerNetworkEvent.Message, this.handleMessage);
            this.peerNetwork.removeListener(IPeerNetwork_1.PeerNetworkEvent.PeersChanged, this.handlePeersChanged);
            yield this.sendLeaveGroupMessage();
            this.started = false;
        });
    }
    sendGroupDataMessage(data) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.started) {
                throw new Error("can't send group message if group not joined");
            }
            const message = {
                type: MessageType.GroupData,
                groupName: this.groupName,
                data,
            };
            this.logDebug('send group_data message', JSON.stringify(message));
            yield this.peerNetwork.send(message);
        });
    }
    addListener(event, callback) {
        this.emitter.addListener(event, callback);
    }
    removeListener(event, callback) {
        this.emitter.removeListener(event, callback);
    }
    removeAllListeners(event) {
        if (event) {
            this.emitter.removeAllListeners(event);
        }
        else {
            this.emitter.removeAllListeners();
        }
    }
    handleMessage(data) {
        return __awaiter(this, void 0, void 0, function* () {
            const { fromId, message } = data;
            if (isQueryMessage(message)) {
                yield this.handleQueryMessage(fromId, message);
            }
            else if (isJoinGroupMessage(message)) {
                this.handleJoinGroupMessage(fromId, message);
            }
            else if (isLeaveGroupMessage(message)) {
                this.handleLeaveGroupMessage(fromId, message);
            }
            else if (isGroupDataMessage(message)) {
                this.handleGroupDataMessage(fromId, message);
            }
        });
    }
    handleQueryMessage(fromId, message) {
        return __awaiter(this, void 0, void 0, function* () {
            if (message.groupName === this.groupName) {
                this.logDebug('got query message from ' + fromId, JSON.stringify(message));
                yield this.sendJoinGroupMessage();
            }
        });
    }
    handleJoinGroupMessage(fromId, message) {
        const { name, groupName, joinedAt } = message;
        if (groupName === this.groupName) {
            const peerAlreadyInList = this.findPeerById(fromId);
            if (!peerAlreadyInList) {
                this.logDebug('got join_group message from ' + fromId, JSON.stringify(message));
                const newMember = {
                    id: fromId,
                    name,
                    joinedAt: new Date(joinedAt),
                };
                this.peers = (0, lodash_1.unionBy)(this.peers, [newMember], 'id'); // ensure unique list
                this.logDebug(`add new member ${fromId}, totalPeers=${this.peers.length}`);
                this.emitter.emit(IGroup_1.GroupEvent.MemberJoined, newMember);
            }
        }
    }
    handleLeaveGroupMessage(fromId, message) {
        const { groupName } = message;
        if (groupName === this.groupName) {
            const groupPeer = this.findPeerById(fromId);
            if (groupPeer) {
                this.logDebug('got leave_group message from ' + fromId, JSON.stringify(message));
                this.peers = this.peers.filter((peer) => peer.id !== fromId);
                this.logDebug(`remove member ${fromId}, totalPeers=${this.peers.length}`);
                this.emitter.emit(IGroup_1.GroupEvent.MemberLeft, groupPeer);
            }
            else {
                this.logDebug(`got leave_group message from ${fromId} but it's not known`, JSON.stringify(message));
            }
        }
    }
    handleGroupDataMessage(fromId, message) {
        if (message.groupName === this.groupName) {
            const groupPeer = this.findPeerById(fromId);
            if (groupPeer) {
                this.logDebug('got group_data message from ' + fromId, JSON.stringify(message));
                this.emitter.emit(IGroup_1.GroupEvent.Data, {
                    from: groupPeer,
                    data: message.data,
                });
            }
            else {
                this.logDebug(`got group_data message from ${fromId} but it's not known`, JSON.stringify(message));
            }
        }
    }
    handlePeersChanged() {
        return __awaiter(this, void 0, void 0, function* () {
            const peers = this.peerNetwork.getPeers();
            this.logDebug('Peers changed', peers);
            yield this.clearDeadPeers();
        });
    }
    findPeerById(id) {
        return this.peers.find((peer) => peer.id === id);
    }
    clearDeadPeers() {
        return __awaiter(this, void 0, void 0, function* () {
            const peers = this.peerNetwork.getPeers();
            const peersIds = peers.map((peer) => peer.id);
            // remove dead peers from the list of peers in the group
            const removedPeers = this.peers.filter((peer) => !peersIds.includes(peer.id));
            this.peers = this.peers.filter((peer) => peersIds.includes(peer.id));
            for (const removedPeer of removedPeers) {
                this.logDebug('peer removed', removedPeer.id);
                this.emitter.emit(IGroup_1.GroupEvent.MemberLeft, removedPeer);
            }
            if (removedPeers.length > 0) {
                this.logDebug(`removed ${removedPeers.length} peers, totalPeers=${this.peers.length}`);
            }
        });
    }
    /**
     * Send query messages periodically to compensate for potential network issues
     * If each peer only sends join_group and query once and someone is unable to receive it
     * at that time because of network issues, they'll never send them again and
     * there will be a partitioned group.
     * Sending query messages periodically ensures that everyone will eventually
     * receive the join_group message from everyone else, as long as the network issues get resolved.
     */
    startSendingQueryMessages() {
        this.sendQueryInterval = setInterval(() => this.sendQueryMessage(), this.sendQueryIntervalMs);
    }
    stopSendingQueryMessages() {
        if (this.sendQueryInterval) {
            clearInterval(this.sendQueryInterval);
            this.sendQueryInterval = null;
        }
    }
    sendQueryMessage() {
        return __awaiter(this, void 0, void 0, function* () {
            const message = {
                type: MessageType.Query,
                groupName: this.groupName,
            };
            this.logDebug('send query message', JSON.stringify(message));
            yield this.peerNetwork.send(message);
        });
    }
    sendJoinGroupMessage() {
        return __awaiter(this, void 0, void 0, function* () {
            const message = {
                type: MessageType.JoinGroup,
                groupName: this.groupName,
                name: this.memberName,
                joinedAt: this.joinedAt.getTime(),
            };
            this.logDebug('send join_group message', JSON.stringify(message));
            yield this.peerNetwork.send(message);
        });
    }
    sendLeaveGroupMessage() {
        return __awaiter(this, void 0, void 0, function* () {
            const message = {
                type: MessageType.LeaveGroup,
                groupName: this.groupName,
            };
            this.logDebug('send leave_group message', JSON.stringify(message));
            yield this.peerNetwork.send(message);
        });
    }
    logDebug(...args) {
        debug(this.groupName, ...args);
    }
}
exports.Group = Group;
__decorate([
    (0, lockedDecorator_1.locked)('join', { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], Group.prototype, "join", null);
//# sourceMappingURL=Group.js.map