"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.socketPlatformCreate = void 0;
const socketActionCreator_1 = require("./socketActionCreator");
const effects_1 = require("redux-saga/effects");
const circular_json_1 = __importDefault(require("circular-json"));
const wait_1 = __importDefault(require("@signageos/lib/dist/Timer/wait"));
const channels_1 = require("../ReduxSaga/channels");
const createSocketIOSocket_1 = __importDefault(require("@signageos/lib/dist/WebSocket/Client/SocketIO/createSocketIOSocket"));
const createWSSocket_1 = require("@signageos/lib/dist/WebSocket/Client/WS/createWSSocket");
const createHttpSocket_1 = require("./Http/createHttpSocket");
const socketActions_1 = require("@signageos/actions/dist/Socket/socketActions");
const applicationActions_1 = require("../Application/applicationActions");
const property_1 = require("../Object/property");
const dateTimeFactory_1 = require("@signageos/lib/dist/DateTime/dateTimeFactory");
const events_1 = require("events");
const socketActions_2 = require("./socketActions");
const serverTimeCheckerSaga_1 = require("../Device/DateTime/serverTimeCheckerSaga");
const debug_1 = __importDefault(require("debug"));
const progressiveWait_1 = require("@signageos/lib/dist/Timer/progressiveWait");
const promise_1 = require("../Util/promise");
const socketActionFilter_1 = require("./socketActionFilter");
const offlineActions_1 = require("../Offline/offlineActions");
const lodash_1 = require("lodash");
const networkActions_1 = require("@signageos/actions/dist/Network/networkActions");
const debug = (0, debug_1.default)('@signageos/front-display:Socket:socketSagas');
const MESSAGE_SERVER_AUTHENTICATE = 'server-authenticate';
const MESSAGE_SERVER_AUTHENTICATE_DONE = 'server-authenticate-done';
const MESSAGE_CLIENT_AUTHENTICATE = 'client-authenticate';
const MESSAGE_ACTION = 'action';
const messagesToServerQueue = [];
let isHttpSocket = false;
const FAST_OFFLINE_BUFFER_LIMIT = 100;
const ACTION_DELIVERY_TIMEOUT_MS = 30e3;
const HTTP_SOCKET_CHECK_INTERVAL_MS = 60e3;
function* socketPlatformCreate(getNativeDriver, socketName, getDeviceTimestamp, defaultConfig, powerKernel) {
    const actionChannel = (0, channels_1.createChannel)();
    const offlineActionChannel = (0, channels_1.createChannel)();
    const fastOfflineBuffer = [];
    const connectionEmitter = new events_1.EventEmitter();
    const progressiveWait = (0, progressiveWait_1.createProgressiveWait)(5e3, 1.5, 5 * 60e3);
    let socketDriver;
    let platformUri;
    let checkTimeBeforeConnection;
    let connected = false;
    let exited = false;
    powerKernel === null || powerKernel === void 0 ? void 0 : powerKernel.doBeforeShutDown(waitUntilTheQueIsEmpty);
    function handleOfflineAction(action) {
        action.__dispatchedAt = action.__dispatchedAt ? action.__dispatchedAt : (0, dateTimeFactory_1.now)().toDate();
        action.__offline = true;
        if (fastOfflineBuffer.length < FAST_OFFLINE_BUFFER_LIMIT && (0, socketActionFilter_1.shouldBeAddedToFastOfflineBuffer)(action)) {
            fastOfflineBuffer.push(action);
        }
        else {
            offlineActionChannel.put(action);
        }
    }
    yield (0, channels_1.takeEvery)(offlineActionChannel, function* (action) {
        if (action.type !== offlineActions_1.SaveOfflineAction) {
            yield (0, effects_1.put)({ type: offlineActions_1.SaveOfflineAction, action });
        }
    });
    function streamAction(action, actionDeliveryTimeoutMs) {
        if (connected) {
            const timeoutHandler = setTimeout(() => streamAction(action, actionDeliveryTimeoutMs), actionDeliveryTimeoutMs);
            action.__emitter.once('delivered', () => clearTimeout(timeoutHandler));
            actionChannel.put(action);
        }
        else if (action.type !== offlineActions_1.SaveOfflineAction) {
            handleOfflineAction(action);
        }
    }
    const boundActions = {};
    yield (0, effects_1.takeEvery)(socketActions_2.BindAction, function* (bindAction) {
        const actionName = bindAction.actionName;
        if (typeof boundActions[actionName] === 'undefined') {
            boundActions[actionName] = 1;
        }
        else {
            boundActions[actionName]++;
        }
    });
    yield (0, effects_1.takeEvery)(socketActions_2.UnbindAction, function* (unbindAction) {
        const actionName = unbindAction.actionName;
        if (boundActions[actionName] === 1) {
            delete boundActions[actionName];
        }
        else if (typeof boundActions[actionName] !== 'undefined') {
            boundActions[actionName]--;
        }
        else {
            debug("unbinding action that wasn't bound: " + actionName);
        }
    });
    yield (0, effects_1.takeEvery)('*', function* (action) {
        const timeout = getActionDeliveryTimeout(socketDriver);
        streamAction(action, timeout);
    });
    yield (0, socketActionCreator_1.bindAndTakeEvery)(socketActions_1.SocketConnected, function* () {
        const timeout = getActionDeliveryTimeout(socketDriver);
        let action;
        while ((action = fastOfflineBuffer.shift())) {
            streamAction(action, timeout);
        }
    });
    yield (0, socketActionCreator_1.bindAndTakeEvery)(applicationActions_1.StopApplication, function* () {
        exited = true;
        connectionEmitter.emit('terminate');
    });
    yield (0, socketActionCreator_1.bindAndTakeEvery)(socketActions_1.PlatformConnect, function* () {
        connectionEmitter.emit('terminate');
    });
    yield (0, socketActionCreator_1.bindAndTakeEvery)(socketActions_1.PlatformConnect, function* (connectAction) {
        socketDriver = connectAction.driver;
        platformUri = connectAction.platformUri;
        checkTimeBeforeConnection = connectAction.checkTimeBeforeConnection;
    });
    yield (0, socketActionCreator_1.bindAndTakeEvery)(networkActions_1.Disconnected, function* () {
        connectionEmitter.emit('terminate');
    });
    yield (0, effects_1.fork)(function* () {
        if (!platformUri) {
            yield (0, effects_1.take)(socketActions_1.PlatformConnect);
        }
        yield (0, wait_1.default)(1e3); // Wait before first socket connect, because device is always open until server configuration say different
        do {
            try {
                yield (0, effects_1.race)([
                    (0, effects_1.call)(serverTimeCheckerSaga_1.checkServerTimeWhenRequested, getNativeDriver, defaultConfig, checkTimeBeforeConnection, getDeviceTimestamp),
                    (0, effects_1.call)(() => (0, promise_1.rejectOnceEvent)(connectionEmitter, 'terminate')),
                ]);
                const duid = yield getNativeDriver().getDeviceUid();
                const createSocket = getSocketFactory(socketDriver);
                const socket = createSocket(platformUri + '/v2/', () => connectionEmitter.emit('connected'), () => connectionEmitter.emit('disconnected'), (error) => connectionEmitter.emit('error', error));
                const connectedPromise = (0, promise_1.resolveOnceEvent)(connectionEmitter, 'connected');
                const disconnectedPromise = (0, promise_1.rejectOnceEvent)(connectionEmitter, 'disconnected');
                const terminatePromise = (0, promise_1.rejectOnceEvent)(connectionEmitter, 'terminate');
                const errorPromise = (0, promise_1.rejectOnceEvent)(connectionEmitter, 'error');
                const socketChannel = createSocketChannel(socket, MESSAGE_ACTION);
                debug('connecting to the socket');
                try {
                    yield Promise.race([
                        connectedPromise,
                        disconnectedPromise, // if unable to connect (reject)
                        errorPromise, // for sure when erred (reject)
                    ]);
                }
                catch (error) {
                    yield (0, effects_1.put)({
                        type: socketActions_2.SocketConnectionFailed,
                        errorMessage: error === null || error === void 0 ? void 0 : error.message,
                    });
                    throw error;
                }
                debug('socket connected');
                yield (0, effects_1.put)({ type: socketActions_2.SocketConnected });
                try {
                    yield Promise.race([
                        disconnectedPromise, // for sure when disconnected
                        errorPromise, // for sure when erred (reject)
                        terminatePromise,
                        authenticateSocket(socket, duid, socketName, Object.keys(boundActions)),
                    ]);
                    debug('socket authenticated');
                    const bindEffect = yield (0, effects_1.fork)(socketBindActions, socketChannel);
                    const emitEffect = yield (0, effects_1.fork)(socketEmitActions, actionChannel, socket, MESSAGE_ACTION);
                    connected = true;
                    yield (0, effects_1.put)({ type: socketActions_1.SocketConnected });
                    progressiveWait.reset();
                    try {
                        yield Promise.race([
                            terminatePromise,
                            disconnectedPromise, // for sure when disconnected
                            errorPromise, // for sure when erred (reject)
                        ]);
                    }
                    finally {
                        connected = false;
                        yield (0, effects_1.cancel)(bindEffect);
                        yield (0, effects_1.cancel)(emitEffect);
                    }
                }
                finally {
                    debug('socket disconnected');
                    yield (0, effects_1.put)({ type: socketActions_1.SocketDisconnected });
                    socket.removeAllListeners();
                    socket.close(); // TODO should be always disconnected so should not be necessary to explicitly close
                }
            }
            catch (error) {
                debug('socketPlatformCreate failed', error);
            }
            finally {
                connectionEmitter.removeAllListeners('error');
                connectionEmitter.removeAllListeners('connected');
                connectionEmitter.removeAllListeners('disconnected');
                connectionEmitter.removeAllListeners('terminate');
            }
            yield progressiveWait.wait();
        } while (!exited);
    });
}
exports.socketPlatformCreate = socketPlatformCreate;
function authenticateSocket(socket, duid, name, actions) {
    return __awaiter(this, void 0, void 0, function* () {
        yield new Promise((resolve) => {
            socket.once(MESSAGE_SERVER_AUTHENTICATE, () => resolve());
        });
        socket.emit(MESSAGE_CLIENT_AUTHENTICATE, { duid, name, actions });
        yield new Promise((resolve) => {
            socket.once(MESSAGE_SERVER_AUTHENTICATE_DONE, () => resolve());
        });
    });
}
function getSocketFactory(socketDriver) {
    switch (socketDriver) {
        case 'ws':
            isHttpSocket = false;
            return createWSSocket_1.createSocket;
        case 'http':
            isHttpSocket = true;
            return (...args) => (0, createHttpSocket_1.createSocket)(...args, {
                messagesToServerQueue,
                checkInterval: HTTP_SOCKET_CHECK_INTERVAL_MS,
            });
        case 'socket.io':
            isHttpSocket = false;
            return createSocketIOSocket_1.default;
        default:
            throw new Error(`Unknown socket driver ${socketDriver}`);
    }
}
function getActionDeliveryTimeout(socketDriver) {
    switch (socketDriver) {
        case 'http':
            return HTTP_SOCKET_CHECK_INTERVAL_MS + ACTION_DELIVERY_TIMEOUT_MS;
        default:
            return ACTION_DELIVERY_TIMEOUT_MS;
    }
}
function* socketEmitActions(actionChannel, socket, messageName) {
    function* takeAction(action) {
        try {
            yield (0, effects_1.call)(emitAction, socket, messageName, action);
        }
        catch (error) {
            debug('emitAction failed', error);
        }
    }
    do {
        const action = yield (0, effects_1.call)(actionChannel.take);
        yield (0, effects_1.fork)(takeAction, action);
    } while (true);
}
function* socketBindActions(socketChannel) {
    function* takeAction(action) {
        yield (0, effects_1.put)(action);
    }
    do {
        const action = yield (0, effects_1.call)(socketChannel.take);
        yield (0, effects_1.fork)(takeAction, action);
    } while (true);
}
function createSocketChannel(socket, messageName) {
    const socketChannel = (0, channels_1.createChannel)();
    socket.on(messageName, (action) => {
        (0, property_1.defineInvisibleProperty)(action, '__source', socket);
        socketChannel.put(action);
    });
    return socketChannel;
}
function emitAction(socket, messageName, action) {
    if (!isFromServer(action, socket) && (0, socketActionFilter_1.shouldBeSentToServer)(action)) {
        return new Promise((resolve, reject) => {
            try {
                const actionMessage = JSON.parse(circular_json_1.default.stringify(action));
                socket.emit(messageName, actionMessage, () => {
                    action.__emitter.emit('delivered');
                    resolve();
                });
            }
            catch (error) {
                reject(error);
            }
        });
    }
    else {
        action.__emitter.emit('delivered');
        return Promise.resolve();
    }
}
function isFromServer(action, socket) {
    return action.__source === socket;
}
function waitUntilTheQueIsEmpty() {
    return __awaiter(this, void 0, void 0, function* () {
        return new Promise((resolve) => {
            if (isHttpSocket) {
                const interval = setInterval(() => {
                    if ((0, lodash_1.isEmpty)(messagesToServerQueue)) {
                        clearInterval(interval);
                        resolve();
                    }
                }, 1e3);
            }
            else {
                resolve();
            }
        });
    });
}
//# sourceMappingURL=socketSagas.js.map