"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSocket = void 0;
const fetch_1 = require("../../Isomorphic/fetch");
const generator_1 = require("@signageos/lib/dist/Hash/generator");
const deferred_1 = require("@signageos/lib/dist/Promise/deferred");
const ISocket_1 = require("@signageos/lib/dist/WebSocket/Client/ISocket");
const debug_1 = __importDefault(require("debug"));
const events_1 = require("events");
const progressiveWait_1 = require("@signageos/lib/dist/Timer/progressiveWait");
const debug = (0, debug_1.default)('@signageos/front-display:Socket:Http:createHttpSocket');
var Resource;
(function (Resource) {
    Resource["SESSION"] = "/http-socket/session";
    Resource["TRANSMIT_MESSAGES"] = "/http-socket/transmit-messages";
    Resource["RECEIVE_MESSAGES"] = "/http-socket/receive-messages";
})(Resource || (Resource = {}));
const DEFAULT_CHECK_INTERVAL = 60e3;
function createSocket(baseUrl, onConnected, onDisconnected, onError, options = {
    messagesToServerQueue: [],
}) {
    var _a;
    const maxCheckInterval = (_a = options.checkInterval) !== null && _a !== void 0 ? _a : DEFAULT_CHECK_INTERVAL;
    const progressiveWait = (0, progressiveWait_1.createProgressiveWait)(Math.min(1e3, maxCheckInterval), 1.5, maxCheckInterval);
    const messageEmitter = new events_1.EventEmitter();
    const responseEmitter = new events_1.EventEmitter();
    const messagesToServerQueue = options.messagesToServerQueue;
    const terminationDeferred = (0, deferred_1.createDeferred)();
    let closed = false;
    let sessionCredentials;
    function emitUnconfirmedMessages() {
        let message;
        while ((message = messagesToServerQueue.shift())) {
            debug('Undelivered message', message);
            if (message.type === 'request' && message.responseUid) {
                const responseMessage = { type: 'response', responseUid: message.responseUid };
                const error = new ISocket_1.UnconfirmedMessageError(responseMessage, message, new Date());
                onError(error);
            }
        }
    }
    function getAuthHeaders() {
        if (sessionCredentials !== undefined) {
            return {
                'X-Auth': sessionCredentials.id + ':' + sessionCredentials.secret,
            };
        }
        throw new Error(`Session has not been initialized yet`);
    }
    function createSession() {
        return __awaiter(this, void 0, void 0, function* () {
            const sessionUri = baseUrl + Resource.SESSION;
            const response = yield (0, fetch_1.fetch)(sessionUri, {
                method: 'POST',
            });
            if (!response.ok) {
                throw new Error(`Request for POST ${Resource.SESSION} failed with status code: ${response.status} - ${response.statusText}`);
            }
            const credentials = yield response.json();
            sessionCredentials = credentials;
        });
    }
    function removeSession() {
        return __awaiter(this, void 0, void 0, function* () {
            if (sessionCredentials === undefined) {
                throw new Error(`Session has not been initialized yet`);
            }
            const sessionUri = baseUrl + Resource.SESSION + '/' + sessionCredentials.id;
            const response = yield (0, fetch_1.fetch)(sessionUri, {
                method: 'DELETE',
                headers: getAuthHeaders(),
            });
            if (!response.ok) {
                throw new Error(`Request for DELETE ${Resource.SESSION}/${sessionCredentials.id} ` +
                    `failed with status code: ${response.status} - ${response.statusText}`);
            }
            sessionCredentials = undefined;
        });
    }
    function processMessagesToServer() {
        return __awaiter(this, void 0, void 0, function* () {
            let message;
            const messagesToTransmit = [];
            while ((message = messagesToServerQueue.shift())) {
                messagesToTransmit.push(message);
            }
            try {
                const messagesUri = baseUrl + Resource.TRANSMIT_MESSAGES;
                const response = yield (0, fetch_1.fetch)(messagesUri, {
                    method: 'POST',
                    headers: Object.assign(Object.assign({}, getAuthHeaders()), { 'Content-Type': 'application/json' }),
                    body: JSON.stringify(messagesToTransmit),
                });
                if (!response.ok) {
                    throw new Error(`Request for POST ${Resource.TRANSMIT_MESSAGES} failed with status code: ${response.status} - ${response.statusText}`);
                }
                for (const messageToTransmit of messagesToTransmit) {
                    if (messageToTransmit.responseUid) {
                        responseEmitter.emit(messageToTransmit.responseUid);
                    }
                }
            }
            catch (error) {
                // Push all messages back to queue in the same order on error
                while ((message = messagesToTransmit.pop())) {
                    messagesToServerQueue.unshift(message);
                }
                throw error;
            }
        });
    }
    function processMessagesFromServer() {
        return __awaiter(this, void 0, void 0, function* () {
            const messagesUri = baseUrl + Resource.RECEIVE_MESSAGES;
            const response = yield (0, fetch_1.fetch)(messagesUri, {
                method: 'POST',
                headers: getAuthHeaders(),
            });
            if (!response.ok) {
                throw new Error(`Request for POST ${Resource.RECEIVE_MESSAGES} failed with status code: ${response.status} - ${response.statusText}`);
            }
            const messages = yield response.json();
            for (const message of messages) {
                messageEmitter.emit(message.event, message.payload);
                if (message.responseUid) {
                    messagesToServerQueue.push({
                        type: 'response',
                        responseUid: message.responseUid,
                    });
                }
            }
            if (messages.length > 0) {
                progressiveWait.reset();
            }
        });
    }
    function doRequesting() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield createSession();
                onConnected();
                while (!closed) {
                    yield processMessagesToServer();
                    yield processMessagesFromServer();
                    yield Promise.race([progressiveWait.wait(), terminationDeferred.promise]);
                }
            }
            finally {
                emitUnconfirmedMessages();
                onDisconnected();
                try {
                    yield removeSession();
                }
                catch (error) {
                    // Clearing session is not required. Will not panic and continue.
                    console.error(`Removing session failed`, error);
                }
            }
        });
    }
    // Run requesting to HTTP in background (do not wait on result)
    doRequesting().catch((error) => console.error(error));
    const socket = {
        on(event, listener) {
            messageEmitter.on(event, listener);
        },
        once(event, listener) {
            messageEmitter.once(event, listener);
        },
        emit(event, payload, callback) {
            const message = { type: 'request', event, payload };
            if (closed) {
                debug('Socket is not open. Undelivered message', message);
                const error = new ISocket_1.UndeliveredEmitError(message, new Date());
                onError(error);
                throw error;
            }
            if (callback) {
                message.responseUid = (0, generator_1.generateUniqueHash)();
                responseEmitter.once(message.responseUid, () => callback());
            }
            messagesToServerQueue.push(message);
        },
        removeListener(event, listener) {
            messageEmitter.removeListener(event, listener);
        },
        removeAllListeners() {
            messageEmitter.removeAllListeners();
        },
        close() {
            if (closed) {
                debug('Http socket is closed already');
                return;
            }
            closed = true;
            terminationDeferred.resolve();
        },
    };
    return socket;
}
exports.createSocket = createSocket;
//# sourceMappingURL=createHttpSocket.js.map