"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PeerDiscoveryServiceMessageSender = void 0;
const events_1 = require("events");
const debug_1 = __importDefault(require("debug"));
const IUdpSocket_1 = require("../Socket/Udp/IUdpSocket");
const debug = (0, debug_1.default)('@signageos/front-display:PeerNetwork:PeerDiscoveryServiceMessageSender');
var MessageType;
(function (MessageType) {
    MessageType["Query"] = "query";
    MessageType["Announce"] = "announce";
    MessageType["Renounce"] = "renounce";
    MessageType["Ping"] = "ping";
})(MessageType || (MessageType = {}));
function isMessage(message) {
    return typeof message === 'object' && message !== null && 'type' in message;
}
function isBaseMessage(message) {
    const messageTypes = Object.keys(MessageType).map((key) => MessageType[key]);
    return isMessage(message) && messageTypes.indexOf(message.type) !== -1;
}
var Event;
(function (Event) {
    Event["Query"] = "query";
    Event["Announce"] = "announce";
    Event["Renounce"] = "renounce";
    Event["Ping"] = "ping";
    Event["Data"] = "data";
})(Event || (Event = {}));
/**
 * Companion class to PeerDiscoveryService that handles sending messages.
 */
class PeerDiscoveryServiceMessageSender {
    constructor(id, udpSocketFactory) {
        this.id = id;
        this.udpSocketFactory = udpSocketFactory;
        this.emitter = new events_1.EventEmitter();
        this.socket = null;
        this.handleMessage = this.handleMessage.bind(this);
    }
    setId(id) {
        this.id = id;
    }
    start() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.socket) {
                throw new Error('PeerDiscoveryServiceMessageSender already started');
            }
            this.socket = yield this.udpSocketFactory();
            this.socket.addListener(IUdpSocket_1.UdpSocketEvent.Message, this.handleMessage);
        });
    }
    stop() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error('PeerDiscoveryServiceMessageSender already stopped');
            }
            yield this.socket.close();
            this.socket.removeListener(IUdpSocket_1.UdpSocketEvent.Message, this.handleMessage);
            this.socket = null;
        });
    }
    isStarted() {
        return Boolean(this.socket);
    }
    onQuery(callback) {
        this.emitter.on(Event.Query, callback);
    }
    sendQuery() {
        return this.sendMessage({
            type: MessageType.Query,
        });
    }
    onAnnounce(callback) {
        this.emitter.on(Event.Announce, callback);
    }
    sendAnnounce(address, port) {
        return this.sendMessage({
            type: MessageType.Announce,
            address,
            port,
        });
    }
    onRenounce(callback) {
        this.emitter.on(Event.Renounce, callback);
    }
    sendRenounce() {
        return this.sendMessage({
            type: MessageType.Renounce,
        });
    }
    onPing(callback) {
        this.emitter.on(Event.Ping, callback);
    }
    sendPing() {
        return this.sendMessage({
            type: MessageType.Ping,
        });
    }
    handleMessage(message) {
        if (message && isBaseMessage(message)) {
            if (message.sourceId === this.id) {
                // ignore own message that come back via broadcast
                return;
            }
            debug('received message', JSON.stringify(message));
            switch (message.type) {
                case MessageType.Query:
                    this.emitter.emit(Event.Query, message);
                    break;
                case MessageType.Announce:
                    this.emitter.emit(Event.Announce, message);
                    break;
                case MessageType.Renounce:
                    this.emitter.emit(Event.Renounce, message);
                    break;
                case MessageType.Ping:
                    this.emitter.emit(Event.Ping, message);
                    break;
                default:
                    console.warn(`UDPPeerDiscoveryService: Invalid message ${JSON.stringify(message)}`);
            }
        }
        else {
            console.warn(`UDPPeerDiscoveryService: Invalid message ${JSON.stringify(message)}`);
        }
    }
    sendMessage(contents) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error("Trying to send message but PeerDiscoveryServiceMessageSender didn't start");
            }
            const message = Object.assign({ sourceId: this.id }, contents);
            debug('send message', JSON.stringify(message));
            yield this.socket.send(message);
        });
    }
}
exports.PeerDiscoveryServiceMessageSender = PeerDiscoveryServiceMessageSender;
//# sourceMappingURL=PeerDiscoveryServiceMessageHandler.js.map