"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const cssRotationHelper_1 = require("./cssRotationHelper");
/**
 * Wrapper class for any implementation of stream player that converts CSS-rotated coordinates back to landscape for the system
 *
 * When the content is rotated via CSS, all the coordinates that are sent from it are rotated as well.
 * That's bad because from the point of view of the system it's landscape and the rotated coordinates don't make sense.
 * This is a proxy class that converts the coordinates between the client and the system so neither of them have to worry about it.
 */
class CSSRotationStreamPlayerAdapter {
    constructor(window, streamPlayer, getAngle) {
        this.window = window;
        this.streamPlayer = streamPlayer;
        this.getAngle = getAngle;
    }
    prepare(uri, x, y, width, height, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.streamPlayer.prepare(uri, coords.x, coords.y, coords.width, coords.height, options);
        });
    }
    play(uri, x, y, width, height, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            const stream = yield this.streamPlayer.play(uri, coords.x, coords.y, coords.width, coords.height, options);
            return this.convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(stream, x, y, width, height);
        });
    }
    stop(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.streamPlayer.stop(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    pause(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.streamPlayer.pause(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    resume(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.streamPlayer.resume(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    clearAll() {
        return __awaiter(this, void 0, void 0, function* () {
            return this.streamPlayer.clearAll();
        });
    }
    getTracks(videoId) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.streamPlayer.getTracks(videoId);
        });
    }
    selectTrack(videoId, trackType, groupId, trackIndex) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.streamPlayer.selectTrack(videoId, trackType, groupId, trackIndex);
        });
    }
    resetTrack(videoId, trackType, groupId) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.streamPlayer.resetTrack(videoId, trackType, groupId);
        });
    }
    convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(streamEmitter, originalX, originalY, originalWidth, originalHeight) {
        const convertedStreamEmitter = new events_1.EventEmitter();
        const convertEvent = (event) => (Object.assign(Object.assign({}, event), { uri: event.uri, x: originalX, y: originalY, width: originalWidth, height: originalHeight, options: event.options }));
        streamEmitter.on('error', (event) => convertedStreamEmitter.emit('error', convertEvent(event)));
        streamEmitter.on('connected', (event) => convertedStreamEmitter.emit('connected', convertEvent(event)));
        streamEmitter.on('disconnected', (event) => convertedStreamEmitter.emit('disconnected', convertEvent(event)));
        streamEmitter.on('closed', (event) => convertedStreamEmitter.emit('closed', convertEvent(event)));
        streamEmitter.on('tracks_changed', (event) => convertedStreamEmitter.emit('tracks_changed', convertEvent(event)));
        // "error" event type is treated as a special case and has to have at least one listener or it can crash the whole process
        // https://nodejs.org/api/events.html#events_error_events
        convertedStreamEmitter.on('error', () => {
            /* do nothing */
        });
        return convertedStreamEmitter;
    }
}
exports.default = CSSRotationStreamPlayerAdapter;
//# sourceMappingURL=CSSRotationStreamPlayerAdapter.js.map