"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cron_1 = require("cron");
const events_1 = require("events");
/**
 * Power Action Scheduler backed by ``CronJob``s which emits ``PowerActionEvent``s.
 */
class CronPowerActionScheduler {
    constructor() {
        this.cronJobs = {};
        this.events = new events_1.EventEmitter();
        this.events.setMaxListeners(1);
    }
    addListener(listener) {
        this.events.addListener('power', listener);
    }
    removeListener(listener) {
        this.events.removeListener('power', listener);
    }
    reschedule(rules) {
        this.unschedule();
        this.schedule(rules);
    }
    unschedule() {
        Object.keys(this.cronJobs).forEach((uid) => {
            this.cronJobs[uid].stop();
            delete this.cronJobs[uid];
        });
    }
    schedule(rules) {
        Object.keys(rules).forEach((uid) => {
            const { powerType, weekdays, time } = rules[uid];
            const cronJob = createCronJob(weekdays, time, () => {
                this.events.emit('power', powerType);
            });
            this.cronJobs[uid] = cronJob;
        });
        Object.keys(this.cronJobs).forEach((uid) => {
            this.cronJobs[uid].start();
        });
    }
}
exports.default = CronPowerActionScheduler;
function createCronJob(weekdays, time, onTick) {
    const weekdaysString = weekdays.join(',');
    const [, hours, minutes, seconds] = (time.match(/^([0-9]{2}):([0-9]{2})(?::([0-9]{2}))?$/) || []);
    return new cron_1.CronJob(`${seconds !== null && seconds !== void 0 ? seconds : '00'} ${minutes} ${hours} * * ${weekdaysString}`, onTick);
}
//# sourceMappingURL=CronPowerActionScheduler.js.map