"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.licenseSaga = void 0;
const applicationActions_1 = require("@signageos/actions/dist/Application/applicationActions");
const effects_1 = require("redux-saga/effects");
const periodicTaskSaga_1 = require("../../Saga/periodicTaskSaga");
const moment_timezone_1 = __importDefault(require("moment-timezone"));
const dateTimeFactory_1 = require("@signageos/lib/dist/DateTime/dateTimeFactory");
const licenseActions_1 = require("./licenseActions");
const Property_1 = __importDefault(require("../../Property/Property"));
const socketActions_1 = require("../../Socket/socketActions");
const debug_1 = __importDefault(require("debug"));
const debug = (0, debug_1.default)('@signageos/front-display:License:licenseSaga');
const INITIAL_DELAY_MS = 120e3; // 120 seconds - to give a time to connect device to web socket for the first time
const PERIOD_MS = 30 * 60e3; // 30 minutes
const SOFT_EXPIRATION_DAYS = 8 * 7; // 8 weeks - this time is in Terms & Conditions of Open product
const HARD_EXPIRATION_DAYS = 16 * 7; // 16 weeks - double the time of T&C to prevent apocalypse
const RESPONSE_TIMEOUT_MS = 60e3;
/**
 * License is taking care of OPEN devices, which has to be connected to our cloud with connection for pinging.
 * Only for open devices, the device can be banned for any signageos operation and showing content.
 * The first soft expiration will show overlay with warning.
 * The second hard expiration will trigger content disabling.
 */
function* licenseSaga(propertyStorage, initialDelayMs = INITIAL_DELAY_MS, periodMs = PERIOD_MS, responseTimeoutMs = RESPONSE_TIMEOUT_MS) {
    let lastSuccessfulPingAt = new Date(0);
    yield (0, effects_1.takeEvery)([socketActions_1.SocketConnected, socketActions_1.CheckSocketHealth], function* (action) {
        try {
            yield new Promise((resolve, reject) => {
                const timeoutHandler = setTimeout(() => reject(new Error('License check action delivery timed out')), responseTimeoutMs);
                action.__emitter.once('delivered', () => {
                    clearTimeout(timeoutHandler);
                    resolve();
                });
            });
            lastSuccessfulPingAt = new Date();
            yield propertyStorage.setValue(Property_1.default.LAST_SUCCESSFUL_PING_AT, lastSuccessfulPingAt.valueOf());
        }
        catch (error) {
            debug('licenseSaga failed', error);
        }
    });
    let lastSeverity = licenseActions_1.ViolationSeverity.none;
    yield (0, effects_1.takeEvery)(licenseActions_1.ChangeLicenseState, function* (action) {
        lastSeverity = action.severity;
    });
    function* adaptWarningState(severity) {
        if (severity !== lastSeverity) {
            yield (0, effects_1.put)({
                type: licenseActions_1.ChangeLicenseState,
                severity,
            });
        }
    }
    yield (0, effects_1.takeEvery)(applicationActions_1.StartApplication, function* () {
        lastSuccessfulPingAt = new Date(yield propertyStorage.getValueOrDefault(Property_1.default.LAST_SUCCESSFUL_PING_AT, lastSuccessfulPingAt.valueOf()));
        yield (0, periodicTaskSaga_1.runPeriodicTaskSagaWhilePlatform)(initialDelayMs, periodMs, function* () {
            yield adaptWarningState(licenseActions_1.ViolationSeverity.none);
        });
        yield (0, periodicTaskSaga_1.runPeriodicTaskSagaWhileOpen)(initialDelayMs, periodMs, function* () {
            if (isExpired(HARD_EXPIRATION_DAYS, lastSuccessfulPingAt)) {
                yield adaptWarningState(licenseActions_1.ViolationSeverity.hard);
            }
            else if (isExpired(SOFT_EXPIRATION_DAYS, lastSuccessfulPingAt)) {
                yield adaptWarningState(licenseActions_1.ViolationSeverity.soft);
            }
            else {
                yield adaptWarningState(licenseActions_1.ViolationSeverity.none);
            }
        });
    });
}
exports.licenseSaga = licenseSaga;
function isExpired(timeoutDays, date) {
    return (0, moment_timezone_1.default)(date).add(timeoutDays, 'days').isBefore((0, dateTimeFactory_1.now)());
}
//# sourceMappingURL=licenseSagas.js.map