"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleVideoMessage = void 0;
const effects_1 = require("redux-saga/effects");
const redux_saga_1 = require("redux-saga");
const appletVideoActions_1 = require("./appletVideoActions");
const EmitVideoEvent_1 = require("./EmitVideoEvent");
const activeAppletActions_1 = require("../activeAppletActions");
const InternalVideoError_1 = __importDefault(require("../Error/InternalVideoError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const videoErrors_1 = require("../../../NativeDevice/Error/videoErrors");
function* handleVideoMessage(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum) {
    switch (data.type) {
        case messageTypePrefix + '.restore':
            return yield handleRestore();
        case messageTypePrefix + '.video.play':
            return yield handleVideoPlay(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum);
        case messageTypePrefix + '.video.prepare':
            return yield handleVideoPrepare(data, nativeDriver, appletUid, timingChecksum);
        case messageTypePrefix + '.video.stop':
            return yield handleVideoStop(data, nativeDriver, appletUid, timingChecksum);
        case messageTypePrefix + '.video.pause':
            return yield handleVideoPause(data, nativeDriver, appletUid, timingChecksum);
        case messageTypePrefix + '.video.resume':
            return yield handleVideoResume(data, nativeDriver, appletUid, timingChecksum);
        default:
            return null;
    }
}
exports.handleVideoMessage = handleVideoMessage;
function* handleRestore() {
    yield (0, effects_1.put)({ type: activeAppletActions_1.ActiveAppletRestore });
    return {};
}
function* handleVideoPlay(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum) {
    let nativeVideo;
    try {
        nativeVideo = (yield nativeDriver.video.play(data.uri, data.x, data.y, data.width, data.height));
    }
    catch (error) {
        throw new InternalVideoError_1.default({
            kind: 'internalVideoErrorWithOrigin',
            message: "Couldn't play the video.",
            code: ErrorCodes_1.default.VIDEO_PLAY_ERROR,
            origin: `play(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    const nativeVideoEndedChannel = (0, redux_saga_1.eventChannel)((putEvent) => {
        const onEnded = (event) => putEvent({ type: 'ended', event });
        const onError = (event) => putEvent({ type: 'error', event });
        const onStopped = (event) => putEvent({ type: 'stopped', event });
        nativeVideo.once('ended', onEnded);
        nativeVideo.once('error', onError);
        nativeVideo.once('stopped', onStopped);
        return () => {
            nativeVideo.removeListener('ended', onEnded);
            nativeVideo.removeListener('error', onError);
            nativeVideo.removeListener('stopped', onStopped);
        };
    }, redux_saga_1.buffers.dropping(1));
    yield (0, effects_1.spawn)(function* () {
        const message = yield (0, effects_1.take)(nativeVideoEndedChannel);
        try {
            const event = message.event;
            yield (0, effects_1.put)({
                type: EmitVideoEvent_1.EmitVideoEvent,
                event: {
                    type: message.type,
                    srcArguments: event.srcArguments,
                    data: event.data,
                },
            });
        }
        catch (error) {
            console.error(messageTypePrefix + '.video.play event sending failed', error);
        }
        finally {
            nativeVideoEndedChannel.close();
        }
    });
    yield (0, effects_1.put)({ type: appletVideoActions_1.ActiveAppletVideoPlay, appletUid, timingChecksum, data });
    return {};
}
function* handleVideoPrepare(data, nativeDriver, appletUid, timingChecksum) {
    if (data.options && typeof data.options.volume !== 'undefined' && (data.options.volume < 0 || data.options.volume > 100)) {
        throw new Error('Volume out of bounds: ' + data.options.volume);
    }
    try {
        yield nativeDriver.video.prepare(data.uri, data.x, data.y, data.width, data.height, data.options);
    }
    catch (error) {
        if (error instanceof videoErrors_1.NoMoreAvailableVideosError) {
            console.warn('video prepare failed because there are no more available video players');
        }
        else {
            throw new InternalVideoError_1.default({
                kind: 'internalVideoErrorWithOrigin',
                message: "Couldn't prepare the video.",
                code: ErrorCodes_1.default.VIDEO_PREPARE_ERROR,
                origin: `prepare(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    }
    yield (0, effects_1.put)({ type: appletVideoActions_1.ActiveAppletVideoPrepare, appletUid, timingChecksum, data });
    return {};
}
function* handleVideoStop(data, nativeDriver, appletUid, timingChecksum) {
    try {
        yield nativeDriver.video.stop(data.uri, data.x, data.y, data.width, data.height);
    }
    catch (error) {
        throw new InternalVideoError_1.default({
            kind: 'internalVideoErrorWithOrigin',
            message: "Couldn't stop the video.",
            code: ErrorCodes_1.default.VIDEO_STOP_ERROR,
            origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    yield (0, effects_1.put)({ type: appletVideoActions_1.ActiveAppletVideoStop, appletUid, timingChecksum, data });
    return {};
}
function* handleVideoPause(data, nativeDriver, appletUid, timingChecksum) {
    try {
        yield nativeDriver.video.pause(data.uri, data.x, data.y, data.width, data.height);
    }
    catch (error) {
        throw new InternalVideoError_1.default({
            kind: 'internalVideoErrorWithOrigin',
            message: "Couldn't pause the video.",
            code: ErrorCodes_1.default.VIDEO_PAUSE_ERROR,
            origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    yield (0, effects_1.put)({ type: appletVideoActions_1.ActiveAppletVideoPause, appletUid, timingChecksum, data });
    return {};
}
function* handleVideoResume(data, nativeDriver, appletUid, timingChecksum) {
    try {
        yield nativeDriver.video.resume(data.uri, data.x, data.y, data.width, data.height);
    }
    catch (error) {
        throw new InternalVideoError_1.default({
            kind: 'internalVideoErrorWithOrigin',
            message: "Couldn't resume the video.",
            code: ErrorCodes_1.default.VIDEO_RESUME_ERROR,
            origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    yield (0, effects_1.put)({ type: appletVideoActions_1.ActiveAppletVideoResume, appletUid, timingChecksum, data });
    return {};
}
function* videoHandler({ messageTypePrefix, data, frontDriver, appletUid, timingChecksum }) {
    return yield handleVideoMessage(messageTypePrefix, data, frontDriver, appletUid, timingChecksum);
}
exports.default = videoHandler;
//# sourceMappingURL=appletVideoHandler.js.map