"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveProtocolType = exports.handleStreamResetTrack = exports.handleStreamSelectTrack = exports.handleStreamGetTracks = exports.handleStreamResume = exports.handleStreamPause = exports.handleStreamStop = exports.handleStreamPlay = exports.handleStreamPrepare = exports.handleStreamMessage = void 0;
const effects_1 = require("redux-saga/effects");
const activeAppletActions_1 = require("@signageos/actions/dist/Applet/activeAppletActions");
const channels_1 = require("../../../ReduxSaga/channels");
const InternalStreamError_1 = __importDefault(require("../Error/InternalStreamError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const videoErrors_1 = require("../../../NativeDevice/Error/videoErrors");
const streamActions_1 = require("../../../Stream/streamActions");
function* handleStreamMessage(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum) {
    switch (data.type) {
        case messageTypePrefix + '.stream.prepare':
            return yield handleStreamPrepare(data, nativeDriver);
        case messageTypePrefix + '.stream.play':
            return yield handleStreamPlay(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum);
        case messageTypePrefix + '.stream.stop':
            return yield handleStreamStop(data, nativeDriver, appletUid, timingChecksum);
        case messageTypePrefix + '.stream.pause':
            return yield handleStreamPause(data, nativeDriver);
        case messageTypePrefix + '.stream.resume':
            return yield handleStreamResume(data, nativeDriver);
        case messageTypePrefix + '.stream.get_tracks':
            return yield handleStreamGetTracks(data, nativeDriver);
        case messageTypePrefix + '.stream.select_track':
            return yield handleStreamSelectTrack(data, nativeDriver);
        case messageTypePrefix + '.stream.reset_track':
            return yield handleStreamResetTrack(data, nativeDriver);
        default:
            return null;
    }
}
exports.handleStreamMessage = handleStreamMessage;
function* handleStreamPrepare(data, nativeDriver) {
    var _a;
    const streamProtocol = resolveProtocolType(data);
    if (typeof streamProtocol !== 'undefined') {
        data.options = Object.assign(Object.assign({}, data.options), { protocol: streamProtocol });
    }
    try {
        yield nativeDriver.stream.prepare(data.uri, data.x, data.y, data.width, data.height, data.options);
    }
    catch (error) {
        if (error instanceof videoErrors_1.NoMoreAvailableVideosError) {
            console.warn('Video prepare failed because there are no more available video players');
        }
        else {
            throw new InternalStreamError_1.default({
                kind: 'internalStreamErrorWithOrigin',
                message: "Couldn't prepare the stream.",
                code: ErrorCodes_1.default.STREAM_PREPARE_ERROR,
                origin: `prepare(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height}` +
                    `${typeof ((_a = data.options) === null || _a === void 0 ? void 0 : _a.protocol) !== 'undefined' ? ', ' + data.options.protocol : ''})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    }
    return {};
}
exports.handleStreamPrepare = handleStreamPrepare;
function* handleStreamPlay(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum) {
    var _a;
    const streamProtocol = resolveProtocolType(data);
    if (typeof streamProtocol !== 'undefined') {
        data.options = Object.assign(Object.assign({}, data.options), { protocol: streamProtocol });
    }
    let stream;
    try {
        stream = (yield nativeDriver.stream.play(data.uri, data.x, data.y, data.width, data.height, data.options));
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't play the stream.",
            code: ErrorCodes_1.default.STREAM_PLAY_ERROR,
            origin: `play(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height}` +
                `${typeof ((_a = data.options) === null || _a === void 0 ? void 0 : _a.protocol) !== 'undefined' ? ', ' + data.options.protocol : ''})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    const streamChannel = (0, channels_1.createChannel)((putEvent) => {
        stream.once('error', (event) => putEvent(event));
        stream.on('connected', (event) => putEvent(event));
        stream.on('disconnected', (event) => putEvent(event));
        stream.on('tracks_changed', (event) => putEvent(event));
    });
    const streamCloseChannel = (0, channels_1.createChannel)((putEvent) => {
        stream.once('closed', (event) => putEvent(event));
    });
    // spawn process that will take stream events and create messages for applet from them
    const eventsTask = yield (0, effects_1.spawn)(channels_1.takeEvery, streamChannel, function* (streamMessage) {
        var _a, _b, _c, _d, _e, _f;
        try {
            switch (streamMessage.type) {
                case 'error':
                    yield (0, effects_1.put)({
                        type: streamActions_1.StreamError,
                        uri: streamMessage.uri,
                        x: streamMessage.x,
                        y: streamMessage.y,
                        width: streamMessage.width,
                        height: streamMessage.height,
                        protocol: (_b = (_a = streamMessage.options) === null || _a === void 0 ? void 0 : _a.protocol) !== null && _b !== void 0 ? _b : streamMessage.protocol,
                        errorMessage: streamMessage.errorMessage,
                    });
                    break;
                case 'connected':
                    yield (0, effects_1.put)({
                        type: streamActions_1.StreamConnected,
                        uri: streamMessage.uri,
                        x: streamMessage.x,
                        y: streamMessage.y,
                        width: streamMessage.width,
                        height: streamMessage.height,
                        protocol: (_d = (_c = streamMessage.options) === null || _c === void 0 ? void 0 : _c.protocol) !== null && _d !== void 0 ? _d : streamMessage.protocol,
                    });
                    break;
                case 'disconnected':
                    yield (0, effects_1.put)({
                        type: streamActions_1.StreamDisconnected,
                        uri: streamMessage.uri,
                        x: streamMessage.x,
                        y: streamMessage.y,
                        width: streamMessage.width,
                        height: streamMessage.height,
                        protocol: (_f = (_e = streamMessage.options) === null || _e === void 0 ? void 0 : _e.protocol) !== null && _f !== void 0 ? _f : streamMessage.protocol,
                    });
                    break;
                case 'tracks_changed':
                    yield (0, effects_1.put)({
                        type: streamActions_1.StreamTracksChanged,
                        uri: streamMessage.uri,
                        x: streamMessage.x,
                        y: streamMessage.y,
                        width: streamMessage.width,
                        height: streamMessage.height,
                        tracks: streamMessage.tracks,
                    });
                    break;
                default:
            }
        }
        catch (error) {
            console.error(messageTypePrefix + '.stream.play event sending failed', error);
        }
    });
    // spawn process that waits until a "closed" event is emitted, cancels eventsTask process and ends itself
    yield (0, effects_1.spawn)(function* () {
        yield (0, effects_1.call)(streamCloseChannel.take);
        yield (0, effects_1.cancel)(yield eventsTask);
    });
    yield (0, effects_1.put)({ type: activeAppletActions_1.ActiveAppletStreamPlay, appletUid, timingChecksum, data });
    return {};
}
exports.handleStreamPlay = handleStreamPlay;
function* handleStreamStop(data, nativeDriver, appletUid, timingChecksum) {
    try {
        yield nativeDriver.stream.stop(data.uri, data.x, data.y, data.width, data.height);
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't stop the stream.",
            code: ErrorCodes_1.default.STREAM_STOP_ERROR,
            origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    yield (0, effects_1.put)({ type: activeAppletActions_1.ActiveAppletStreamStop, appletUid, timingChecksum, data });
    return {};
}
exports.handleStreamStop = handleStreamStop;
function* handleStreamPause(data, nativeDriver) {
    try {
        yield nativeDriver.stream.pause(data.uri, data.x, data.y, data.width, data.height);
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't pause the stream.",
            code: ErrorCodes_1.default.STREAM_PAUSE_ERROR,
            origin: `pause(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
exports.handleStreamPause = handleStreamPause;
// TODO Is the same extra handling as in handleStreamPlay needed here?
function* handleStreamResume(data, nativeDriver) {
    try {
        yield nativeDriver.stream.resume(data.uri, data.x, data.y, data.width, data.height);
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't resume the stream.",
            code: ErrorCodes_1.default.STREAM_RESUME_ERROR,
            origin: `resume(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
exports.handleStreamResume = handleStreamResume;
function* handleStreamGetTracks(data, nativeDriver) {
    try {
        const tracks = yield nativeDriver.stream.getTracks(data.videoId);
        return { tracks };
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't get stream track info.",
            code: ErrorCodes_1.default.STREAM_GET_TRACKS_ERROR,
            origin: `getTracks(${JSON.stringify(data.videoId)})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
exports.handleStreamGetTracks = handleStreamGetTracks;
function* handleStreamSelectTrack(data, nativeDriver) {
    try {
        yield nativeDriver.stream.selectTrack(data.videoId, data.trackType, data.groupId, data.trackIndex);
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't select stream track.",
            code: ErrorCodes_1.default.STREAM_SELECT_TRACK,
            origin: `selectTrack(${JSON.stringify(data.videoId)}, ${data.trackType}, ${data.groupId}, ${data.trackIndex})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
exports.handleStreamSelectTrack = handleStreamSelectTrack;
function* handleStreamResetTrack(data, nativeDriver) {
    try {
        yield nativeDriver.stream.resetTrack(data.videoId, data.trackType, data.groupId);
    }
    catch (error) {
        throw new InternalStreamError_1.default({
            kind: 'internalStreamErrorWithOrigin',
            message: "Couldn't reset stream track.",
            code: ErrorCodes_1.default.STREAM_RESET_TRACK,
            origin: `resetTrack(${JSON.stringify(data.videoId)}, ${data.trackType}, ${data.groupId})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
exports.handleStreamResetTrack = handleStreamResetTrack;
function resolveProtocolType(data) {
    var _a;
    // Legacy format unused formats (until front-applet 6.x)
    if (data.protocolType !== undefined) {
        return data.protocolType.toUpperCase();
    }
    if (data.protocol !== undefined) {
        return data.protocol.toUpperCase();
    }
    // New format from (front-applet 6.x), used in prepare and play functions
    return (_a = data.options) === null || _a === void 0 ? void 0 : _a.protocol;
}
exports.resolveProtocolType = resolveProtocolType;
function* streamHandler({ messageTypePrefix, data, frontDriver, appletUid, timingChecksum }) {
    return yield handleStreamMessage(messageTypePrefix, data, frontDriver, appletUid, timingChecksum);
}
exports.default = streamHandler;
//# sourceMappingURL=appletStreamHandler.js.map