"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleHardwareSerialPortMessage = void 0;
const effects_1 = require("redux-saga/effects");
const generator_1 = require("@signageos/lib/dist/Hash/generator");
const channels_1 = require("../../../ReduxSaga/channels");
const ISerial_1 = require("../../../NativeDevice/Hardware/ISerial");
const InternalHardwareError_1 = __importDefault(require("../Error/InternalHardwareError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const sendAppletMessage_1 = require("../sendAppletMessage");
function* handleHardwareSerialPortMessage(messageTypePrefix, data, window, nativeDriver, getState) {
    switch (data.type) {
        case messageTypePrefix + '.hardware.serial_port.open':
            return yield openSerialPort(messageTypePrefix, window, nativeDriver, getState, data);
        case messageTypePrefix + '.hardware.serial_port.close':
            return yield closeSerialPort(data);
        case messageTypePrefix + '.hardware.serial_port.write':
            return yield writeToSerialPort(data);
        default:
            return null;
    }
}
exports.handleHardwareSerialPortMessage = handleHardwareSerialPortMessage;
/*
 * Open serial ports have to be persisted between applet messages because later messages reference serial port instances open
 * by earlier messages.
 * Because these message handler are functional and don't have a state, it's a fundamental conflict of interest.
 * A clean solution would be to pass some serial port store object from the top but I decided not to do it, since it would
 * change a lot of code for a very small gain.
 * For now, having this global variable is good enough. I'm aware that it's ugly. If it becomes a problem, we'll refactor it.
 */
const serialPorts = {};
function getDefaultSerialDevice(nativeDriver) {
    switch (nativeDriver.getApplicationType().toLowerCase()) {
        case 'tizen':
            return 'PORT1';
        case 'brightsign':
            return '0';
        case 'windows':
            return 'COM3';
        case 'default':
        case 'linux':
            return '/dev/ttyUSB0';
        case 'android':
            return '/dev/ttyusb0';
        default:
            return null;
    }
}
function* openSerialPort(messageTypePrefix, window, nativeDriver, getState, data) {
    let serialPort;
    if (data.options.device === undefined) {
        const defaultDevice = getDefaultSerialDevice(nativeDriver);
        if (defaultDevice === null) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareError',
                message: 'Failed to get default serial port for device. Please specify manually the device in options object.',
                code: ErrorCodes_1.default.HARDWARE_DEFAULT_DEVICE_NOT_FOUND,
            });
        }
        data.options.device = defaultDevice;
    }
    try {
        serialPort = yield nativeDriver.hardware.serial.openPort(data.options);
    }
    catch (error) {
        throw new InternalHardwareError_1.default({
            kind: 'internalHardwareErrorWithOrigin',
            message: 'Failed to open serial port',
            code: ErrorCodes_1.default.HARDWARE_OPEN_SERIAL_PORT_ERROR,
            origin: `openSerialPort(${JSON.stringify(data.options)})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    const refid = (0, generator_1.generateUniqueHash)(10);
    serialPorts[refid] = serialPort;
    yield sendSerialPortDataToActiveApplet(messageTypePrefix, window, getState, refid, serialPort);
    return { refid };
}
function* sendSerialPortDataToActiveApplet(messageTypePrefix, window, getState, refid, serialPort) {
    const serialPortDataChannel = (0, channels_1.createChannel)((putData) => {
        serialPort.on(ISerial_1.SerialPortEvent.DATA, (data) => {
            putData(data);
        });
    });
    const serialPortCloseChannel = (0, channels_1.createChannel)((putClosed) => {
        serialPort.on(ISerial_1.SerialPortEvent.CLOSE, () => putClosed(true));
    });
    const dataTask = yield (0, effects_1.spawn)(channels_1.takeEvery, serialPortDataChannel, function* (data) {
        yield (0, sendAppletMessage_1.sendMessageToActiveAppletIfExists)(window, getState, {
            type: messageTypePrefix + '.hardware.serial_port.data',
            refid,
            data,
        });
    });
    yield (0, effects_1.spawn)(function* () {
        yield (0, effects_1.call)(serialPortCloseChannel.take);
        yield (0, effects_1.cancel)(dataTask);
        delete serialPorts[refid];
    });
}
function closeSerialPort(data) {
    return __awaiter(this, void 0, void 0, function* () {
        const { refid } = data;
        const serialPort = serialPorts[refid];
        if (!serialPort) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareError',
                message: "Closing serial port that isn't open",
                code: ErrorCodes_1.default.HARDWARE_CLOSE_SERIAL_PORT_ERROR,
            });
        }
        try {
            yield serialPort.close();
        }
        catch (error) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareErrorWithOrigin',
                message: 'Failed to close serial port',
                code: ErrorCodes_1.default.HARDWARE_CLOSE_SERIAL_PORT_ERROR,
                origin: `closeSerialPort(${refid})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        return {};
    });
}
function writeToSerialPort(data) {
    return __awaiter(this, void 0, void 0, function* () {
        const { refid } = data;
        const serialPort = serialPorts[refid];
        if (!serialPort) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareError',
                message: "Writting to a serial port that isn't open",
                code: ErrorCodes_1.default.HARDWARE_WRITE_TO_SERIAL_PORT_ERROR,
            });
        }
        try {
            yield serialPort.write(data.data);
        }
        catch (error) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareErrorWithOrigin',
                message: 'Failed to write to serial port',
                code: ErrorCodes_1.default.HARDWARE_WRITE_TO_SERIAL_PORT_ERROR,
                origin: `writeToSerialPort(${refid})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        return {};
    });
}
function* hardwareSerialPortHandler({ messageTypePrefix, data, frontDriver, window, getState, }) {
    return yield handleHardwareSerialPortMessage(messageTypePrefix, data, window, frontDriver, getState);
}
exports.default = hardwareSerialPortHandler;
//# sourceMappingURL=appletHardwareSerialPortHandler.js.map