"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BarcodeScannerStatus = exports.DEFAULT_SCANNER_ID = void 0;
const InternalHardwareError_1 = __importDefault(require("../Error/InternalHardwareError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const sendAppletMessage_1 = require("../sendAppletMessage");
exports.DEFAULT_SCANNER_ID = 0;
var BarcodeScannerStatus;
(function (BarcodeScannerStatus) {
    BarcodeScannerStatus[BarcodeScannerStatus["IDLE"] = 0] = "IDLE";
    BarcodeScannerStatus[BarcodeScannerStatus["STARTING"] = 1] = "STARTING";
    BarcodeScannerStatus[BarcodeScannerStatus["USED"] = 2] = "USED";
    BarcodeScannerStatus[BarcodeScannerStatus["STOPPING"] = 3] = "STOPPING";
})(BarcodeScannerStatus || (exports.BarcodeScannerStatus = BarcodeScannerStatus = {}));
function handleHardwareBarcodeScannerMessage(scanners, messageTypePrefix, msg, nativeDriver, window, getState) {
    const getMessage = (name) => `${messageTypePrefix}.hardware.barcode_scanner.${name}`;
    switch (msg.type) {
        case getMessage('get_version'):
            return getVersion(nativeDriver);
        case getMessage('start'):
            const startMsg = msg;
            return startScanning(scanners, nativeDriver, window, getState, getMessage, startMsg.scannerId, startMsg.options);
        case getMessage('stop'):
            const stopMsg = msg;
            return stopScanning(scanners, nativeDriver, stopMsg.scannerId);
        default:
            return Promise.resolve(null);
    }
}
function getVersion(nativeDriver) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const version = yield nativeDriver.hardware.barcodeScanner.getVersion();
            return { version };
        }
        catch (error) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareErrorWithOrigin',
                message: 'Failed to get version',
                code: ErrorCodes_1.default.HARDWARE_GET_BARCODE_SCANNER_VERSION_ERROR,
                origin: 'getVersion()',
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function startScanning(scanners, nativeDriver, window, getState, getMessage, scannerId, options) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!scanners[scannerId]) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareError',
                message: 'Selected barcode scanner is unknown',
                code: ErrorCodes_1.default.HARDWARE_START_OF_UNKNOWN_BARCODE_SCANNER,
            });
        }
        else if (scanners[scannerId].status === BarcodeScannerStatus.USED && options.cancelPrevious) {
            yield stopScanning(scanners, nativeDriver, scannerId);
        }
        else if (scanners[scannerId].status !== BarcodeScannerStatus.IDLE) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareError',
                message: 'Selected barcode scanner is busy',
                code: ErrorCodes_1.default.HARDWARE_START_OF_BUSY_BARCODE_SCANNER,
            });
        }
        scanners[scannerId].status = BarcodeScannerStatus.STARTING;
        try {
            const onData = (data) => {
                (0, sendAppletMessage_1.sendMessageToActiveAppletIfExists)(window, getState, {
                    type: getMessage('data'),
                    scannerId,
                    data,
                });
            };
            const onError = (message) => {
                (0, sendAppletMessage_1.sendMessageToActiveAppletIfExists)(window, getState, {
                    type: getMessage('error'),
                    scannerId,
                    data: message,
                });
            };
            yield nativeDriver.hardware.barcodeScanner.startScanning(scannerId, options.timeout, onData, onError);
        }
        catch (error) {
            scanners[scannerId].status = BarcodeScannerStatus.IDLE;
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareErrorWithOrigin',
                message: 'Failed to start scanning barcode',
                code: ErrorCodes_1.default.HARDWARE_START_OF_BARCODE_SCANNER_ERROR,
                origin: `startScanning(${JSON.stringify(options)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        scanners[scannerId] = { status: BarcodeScannerStatus.USED };
        return {};
    });
}
function stopScanning(scanners, nativeDriver, scannerId) {
    return __awaiter(this, void 0, void 0, function* () {
        if (scanners[scannerId].status !== BarcodeScannerStatus.USED) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareError',
                message: 'Barcode scanner to stop is already idle',
                code: ErrorCodes_1.default.HARDWARE_STOP_OF_IDLE_BARCODE_SCANNER,
            });
        }
        scanners[scannerId].status = BarcodeScannerStatus.STOPPING;
        try {
            yield nativeDriver.hardware.barcodeScanner.stopScanning(scannerId);
        }
        catch (error) {
            throw new InternalHardwareError_1.default({
                kind: 'internalHardwareErrorWithOrigin',
                message: 'Failed to stop barcode scanner',
                code: ErrorCodes_1.default.HARDWARE_STOP_OF_BARCODE_SCANNER_ERROR,
                origin: `stopScanning(${scannerId})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        scanners[scannerId].status = BarcodeScannerStatus.IDLE;
        return {};
    });
}
const defaultScanners = { [exports.DEFAULT_SCANNER_ID]: { status: BarcodeScannerStatus.IDLE } };
function* hardwareBarcodeScannerHandler({ messageTypePrefix, data, frontDriver, window, getState, }, mockScanners) {
    const scanners = mockScanners !== null && mockScanners !== void 0 ? mockScanners : defaultScanners;
    return yield handleHardwareBarcodeScannerMessage(scanners, messageTypePrefix, data, frontDriver, window, getState);
}
exports.default = hardwareBarcodeScannerHandler;
//# sourceMappingURL=appletHardwareBarcodeScannerHandler.js.map