"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleFileSystemMessage = void 0;
const InternalFileSystemError_1 = __importDefault(require("../Error/InternalFileSystemError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const basicErrors_1 = require("../../../NativeDevice/Error/basicErrors");
function* handleFileSystemMessage(messageTypePrefix, data, fileSystem) {
    switch (data.type) {
        case messageTypePrefix + '.file_system.list_storage_units':
            return yield handleListStorageUnits(fileSystem);
        case messageTypePrefix + '.file_system.list_files':
            return yield handleListFiles(fileSystem, data);
        case messageTypePrefix + '.file_system.exists':
            return yield handleFileExists(fileSystem, data);
        case messageTypePrefix + '.file_system.get_file':
            return yield handleGetFile(fileSystem, data);
        case messageTypePrefix + '.file_system.write_file':
            return yield handleWriteFile(fileSystem, data);
        case messageTypePrefix + '.file_system.append_file':
            return yield handleAppendFile(fileSystem, data);
        case messageTypePrefix + '.file_system.read_file':
            return yield handleReadFile(fileSystem, data);
        case messageTypePrefix + '.file_system.copy_file':
            return yield handleCopyFile(fileSystem, data);
        case messageTypePrefix + '.file_system.move_file':
            return yield handleMoveFile(fileSystem, data);
        case messageTypePrefix + '.file_system.delete_file':
            return yield handleDeleteFile(fileSystem, data);
        case messageTypePrefix + '.file_system.download_file':
            return yield handleDownloadFile(fileSystem, data);
        case messageTypePrefix + '.file_system.extract_file':
            return yield handleExtractFile(fileSystem, data);
        case messageTypePrefix + '.file_system.link':
            return yield handleLink(fileSystem, data);
        case messageTypePrefix + '.file_system.get_file_checksum':
            return yield handleGetFileChecksum(fileSystem, data);
        case messageTypePrefix + '.file_system.create_directory':
            return yield handleCreateDirectory(fileSystem, data);
        case messageTypePrefix + '.file_system.is_directory':
            return yield handleIsDirectory(fileSystem, data);
        case messageTypePrefix + '.file_system.create_archive_file':
            return yield handleCreateArchive(fileSystem, data);
        case messageTypePrefix + '.file_system.wipeout':
            return yield handleWipeout(fileSystem);
        default:
            return null;
    }
}
exports.handleFileSystemMessage = handleFileSystemMessage;
function humanizeFilePath(filePath) {
    return `storageUnit=${filePath.storageUnit.type},filePath=${filePath.filePath}`;
}
function handleListStorageUnits(fileSystem) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const storageUnits = yield fileSystem.listStorageUnits();
            return { storageUnits };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when listing storage units.',
                code: ErrorCodes_1.default.FILE_SYSTEM_LIST_STORAGE_UNITS_ERROR,
                origin: 'listStorageUnits()',
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleListFiles(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const files = yield fileSystem.listFiles(data.directoryPath);
            return { files };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when listing files.',
                code: ErrorCodes_1.default.FILE_SYSTEM_LIST_FILES_ERROR,
                origin: `listFiles(${humanizeFilePath(data.directoryPath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleFileExists(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const exists = yield fileSystem.exists(data.filePath);
            return { exists };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when checking for file existence.',
                code: ErrorCodes_1.default.FILE_SYSTEM_FILE_EXISTS_ERROR,
                origin: `exists(${humanizeFilePath(data.filePath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleGetFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const file = yield fileSystem.getFile(data.filePath);
            return { file };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when getting a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_GET_FILE_ERROR,
                origin: `getFile(${humanizeFilePath(data.filePath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleWriteFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.writeFile(data.filePath, data.contents);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when writing a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_WRITE_FILE_ERROR,
                origin: `writeFile(${humanizeFilePath(data.filePath)}, ${data.contents.substr(0, 100)}...)`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleAppendFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.appendFile(data.filePath, data.contents);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when writing a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_APPEND_FILE_ERROR,
                origin: `appendFile(${humanizeFilePath(data.filePath)}, ${data.contents.substr(0, 100)}...)`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleReadFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const contents = yield fileSystem.readFile(data.filePath);
            return { contents };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when reading a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_READ_FILE_ERROR,
                origin: `readFile(${humanizeFilePath(data.filePath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleCopyFile(fileSystem, data) {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.copyFile(data.sourceFilePath, data.destinationFilePath, data.options);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when copying a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_COPY_FILE_ERROR,
                origin: `copyFile(${humanizeFilePath(data.sourceFilePath)}, ${humanizeFilePath(data.destinationFilePath)}, ` +
                    `{overwrite:${(_a = data.options) === null || _a === void 0 ? void 0 : _a.overwrite}})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleMoveFile(fileSystem, data) {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.moveFile(data.sourceFilePath, data.destinationFilePath, data.options);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when moving a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_MOVE_FILE_ERROR,
                origin: `moveFile(${humanizeFilePath(data.sourceFilePath)}, ${humanizeFilePath(data.destinationFilePath)}), ` +
                    `{overwrite:${(_a = data.options) === null || _a === void 0 ? void 0 : _a.overwrite}})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleDeleteFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.deleteFile(data.filePath, data.recursive);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when deleting a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_DELETE_FILE_ERROR,
                origin: `deleteFile(${humanizeFilePath(data.filePath)}, ${data.recursive})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleDownloadFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.downloadFile(data.filePath, data.sourceUri, data.headers);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when downloading a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_DOWNLOAD_FILE_ERROR,
                origin: `downloadFile(${humanizeFilePath(data.filePath)}, ${data.sourceUri}, ${JSON.stringify(data.headers)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleExtractFile(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.extractFile(data.archiveFilePath, data.destinationDirectoryPath, data.method);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when extracting a file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_EXTRACT_FILE_ERROR,
                origin: `extractFile(${humanizeFilePath(data.archiveFilePath)}, ${humanizeFilePath(data.destinationDirectoryPath)}, ` + `${data.method})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleCreateArchive(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.createArchive(data.archiveFilePath, data.archiveEntries);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when creating an archive file.',
                code: ErrorCodes_1.default.FILE_SYSTEM_CREATE_ARCHIVE_FILE_ERROR,
                origin: `createArchive(${humanizeFilePath(data.archiveFilePath)}, ${JSON.stringify(data.archiveEntries)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleLink(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.link(data.sourceFilePath, data.destinationFilePath);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: error instanceof basicErrors_1.NotSupportedMethodError
                    ? 'Link is not supported on this platform.'
                    : 'Unexpected error occurred when creating a link.',
                code: error instanceof basicErrors_1.NotSupportedMethodError ? ErrorCodes_1.default.FILE_SYSTEM_LINK_NOT_SUPPORTED : ErrorCodes_1.default.FILE_SYSTEM_LINK_ERROR,
                origin: `link(${humanizeFilePath(data.sourceFilePath)}, ${humanizeFilePath(data.destinationFilePath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleGetFileChecksum(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const checksum = yield fileSystem.getFileChecksum(data.filePath, data.hashType);
            return { checksum };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when getting file checksum.',
                code: ErrorCodes_1.default.FILE_SYSTEM_GET_CHECKSUM_ERROR,
                origin: `getFileChecksum(${humanizeFilePath(data.filePath)}, ${data.hashType})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleCreateDirectory(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.createDirectory(data.directoryPath);
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when creating directory.',
                code: ErrorCodes_1.default.FILE_SYSTEM_CREATE_DIR_ERROR,
                origin: `createDirectory(${humanizeFilePath(data.directoryPath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleIsDirectory(fileSystem, data) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const isDirectory = yield fileSystem.isDirectory(data.filePath);
            return { isDirectory };
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when checking if a path is directory.',
                code: ErrorCodes_1.default.FILE_SYSTEM_IS_DIR_ERROR,
                origin: `isDirectory(${humanizeFilePath(data.filePath)})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleWipeout(fileSystem) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield fileSystem.wipeout();
            return {};
        }
        catch (error) {
            throw new InternalFileSystemError_1.default({
                kind: 'internalFileSystemErrorWithOrigin',
                message: 'Unexpected error occurred when wiping out the file system.',
                code: ErrorCodes_1.default.FILE_SYSTEM_WIPEOUT_ERROR,
                origin: 'wipeout()',
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function* fileSystemHandler({ messageTypePrefix, data, frontDriver }) {
    return yield handleFileSystemMessage(messageTypePrefix, data, frontDriver.fileSystem);
}
exports.default = fileSystemHandler;
//# sourceMappingURL=appletFileSystemHandler.js.map