"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StubMemoryFileSystem = exports.mockFetch = exports.ERRORS = exports.EXISTING_FILE_CHECKSUM = exports.EXISTING_FILE_CONTENT = exports.EXISTING_FILE_REMOTE_URI = exports.EXISTING_FILE_NAME = exports.FAKE_INTERNAL_STORAGE_UNIT = void 0;
const memfs_1 = require("memfs");
const path_1 = __importDefault(require("path"));
const events_1 = require("events");
const fileSystemHelper_1 = require("./fileSystemHelper");
const buffer_1 = require("buffer");
const checksum_1 = require("../Hash/checksum");
const capacity = 20 * 1024 * 1024; // 20 MBi
exports.FAKE_INTERNAL_STORAGE_UNIT = {
    type: 'internal',
    capacity: capacity,
    freeSpace: capacity * 0.875,
    usableSpace: capacity * 0.875,
    removable: false,
};
exports.EXISTING_FILE_NAME = 'storage/file.txt';
exports.EXISTING_FILE_REMOTE_URI = 'https://example.com/file.txt';
exports.EXISTING_FILE_CONTENT = 'some content';
exports.EXISTING_FILE_CHECKSUM = {
    crc32: '431f313f',
    md5: '9893532233caff98cd083a116b013c0b',
};
exports.ERRORS = {
    fileNotFound: 'File not found',
    pathIsNotDirectory: 'Path is not a directory',
    pathIsNotFile: 'Path is not a file',
};
/* c8 ignore start */
function mockFetch(sourceUri, _headers) {
    return __awaiter(this, void 0, void 0, function* () {
        if (sourceUri === exports.EXISTING_FILE_REMOTE_URI) {
            return {
                status: 200,
                ok: true,
                text: () => Promise.resolve(exports.EXISTING_FILE_CONTENT),
                blob: () => Promise.resolve(new buffer_1.Blob([exports.EXISTING_FILE_CONTENT])),
            };
        }
        else {
            return {
                status: 404,
                ok: false,
                text: () => Promise.resolve(''),
                blob: () => Promise.resolve(new buffer_1.Blob([])),
            };
        }
    });
}
exports.mockFetch = mockFetch;
/* c8 ignore stop */
class StubMemoryFileSystem {
    constructor() {
        this.volume = new memfs_1.Volume();
        this.eventEmitter = new events_1.EventEmitter();
    }
    listStorageUnits() {
        return __awaiter(this, void 0, void 0, function* () {
            return [exports.FAKE_INTERNAL_STORAGE_UNIT];
        });
    }
    onStorageUnitsChanged(_listener) {
        // do nothing
    }
    initialize() {
        return __awaiter(this, void 0, void 0, function* () {
            this.volume.mkdirpSync('/storage');
        });
    }
    listFiles(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const exists = yield this.exists(directoryPath);
            if (!exists) {
                throw new Error(exports.ERRORS.fileNotFound);
            }
            const isDirectory = yield this.isDirectory(directoryPath);
            if (!isDirectory) {
                throw new Error(exports.ERRORS.pathIsNotDirectory);
            }
            const absolutePath = this.getAbsolutePath(directoryPath);
            const filenames = this.volume.readdirSync(absolutePath);
            return filenames.map((filename) => ({
                storageUnit: directoryPath.storageUnit,
                filePath: (0, fileSystemHelper_1.trimSlashesAndDots)(path_1.default.join(directoryPath.filePath, filename)),
            }));
        });
    }
    getFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const fileMetadata = this.volume.lstatSync(this.getAbsolutePath(filePath));
                if (!fileMetadata || !fileMetadata.isFile()) {
                    return null;
                }
                const blob = yield this.readFileAsBlob(filePath);
                const file = {
                    localUri: filePath.filePath,
                    mimeType: blob.type,
                    sizeBytes: blob.size,
                    createdAt: fileMetadata.birthtimeMs,
                    lastModifiedAt: fileMetadata.mtimeMs,
                };
                return Object.assign(Object.assign({}, file), filePath);
            }
            catch (error) {
                return null;
            }
        });
    }
    readFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const blob = yield this.readFileAsBlob(filePath);
            const contents = yield blob.text();
            return contents;
        });
    }
    readFileAsBlob(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const stats = this.volume.lstatSync(this.getAbsolutePath(filePath));
            if (!stats.isFile()) {
                throw new Error(exports.ERRORS.pathIsNotFile);
            }
            const absolutePath = this.getAbsolutePath(filePath);
            const contents = this.volume.readFileSync(absolutePath);
            return new buffer_1.Blob([contents]);
        });
    }
    writeFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const fullPath = this.getAbsolutePath(filePath);
            this.volume.writeFileSync(fullPath, contents);
        });
    }
    appendFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const fullPath = this.getAbsolutePath(filePath);
            this.volume.appendFileSync(fullPath, contents);
        });
    }
    exists(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fullPath = this.getAbsolutePath(filePath);
            return this.volume.existsSync(fullPath);
        });
    }
    downloadFile(filePath, sourceUri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield mockFetch(sourceUri, headers);
            if (!response.ok) {
                throw new Error(`Not OK response during downloadFile ${sourceUri}`);
            }
            const blob = yield response.blob();
            yield this.writeBlobAsFile(filePath, blob);
        });
    }
    writeBlobAsFile(filePath, blob) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve) => {
                const fullPath = this.getAbsolutePath(filePath);
                this.volume.writeFileSync(fullPath, blob.toString());
                resolve();
            });
        });
    }
    uploadFile(filePath, uri, formKey, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            let response = '';
            const responseCallback = (thisResponse) => {
                response = thisResponse;
            };
            this.eventEmitter.emit('upload', filePath, uri, formKey, headers, responseCallback);
            return response;
        });
    }
    onUpload(listener) {
        this.eventEmitter.on('upload', listener);
    }
    deleteFile(filePath, recursive) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.exists(filePath))) {
                if (recursive) {
                    return;
                }
                else {
                    throw new Error(exports.ERRORS.fileNotFound);
                }
            }
            const fullPath = this.getAbsolutePath(filePath);
            if (yield this.isDirectory(filePath)) {
                if (recursive) {
                    yield this.deleteFileRecursive(filePath);
                }
                else {
                    this.volume.rmdirSync(fullPath);
                }
            }
            else {
                this.volume.unlinkSync(fullPath);
            }
        });
    }
    copyFile(sourceFilePath, destinationFilePath, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const sourceFullPath = this.getAbsolutePath(sourceFilePath);
            const destinationFullPath = this.getAbsolutePath(destinationFilePath);
            if (this.volume.existsSync(destinationFullPath)) {
                if (options === null || options === void 0 ? void 0 : options.overwrite) {
                    this.volume.unlinkSync(destinationFullPath);
                }
                else {
                    throw new Error('Destination file already exists');
                }
            }
            this.volume.copyFileSync(sourceFullPath, destinationFullPath);
        });
    }
    moveFile(sourceFilePath, destinationFilePath, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const sourceFullPath = this.getAbsolutePath(sourceFilePath);
            const destinationFullPath = this.getAbsolutePath(destinationFilePath);
            if (this.volume.existsSync(destinationFullPath)) {
                if (options === null || options === void 0 ? void 0 : options.overwrite) {
                    this.volume.unlinkSync(destinationFullPath);
                }
                else {
                    throw new Error('Destination file already exists');
                }
            }
            this.volume.renameSync(sourceFullPath, destinationFullPath);
        });
    }
    link(sourceFilePath, destinationFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const sourceFullPath = this.getAbsolutePath(sourceFilePath);
            const destinationFullPath = this.getAbsolutePath(destinationFilePath);
            this.volume.linkSync(sourceFullPath, destinationFullPath);
        });
    }
    getFileChecksum(filePath, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            const blob = yield this.readFileAsBlob(filePath);
            return (0, checksum_1.getChecksumOfBlob)(blob, hashType);
        });
    }
    extractFile(_archiveFilePath, _destinationDirectoryPath, _method) {
        return __awaiter(this, void 0, void 0, function* () {
            // TODO implement when needed
            throw new Error('Method not implemented.');
        });
    }
    createArchive(_archiveFilePath, _archiveEntries) {
        return __awaiter(this, void 0, void 0, function* () {
            throw new Error('Method not implemented.');
        });
    }
    getArchiveInfo(_archiveFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            throw new Error('Method not implemented.');
        });
    }
    wipeout() {
        return __awaiter(this, void 0, void 0, function* () {
            return Promise.resolve();
        });
    }
    createDirectory(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fullPath = this.getAbsolutePath(directoryPath);
            this.volume.mkdirSync(fullPath);
        });
    }
    isDirectory(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.exists(filePath))) {
                return false;
            }
            const fullPath = this.getAbsolutePath(filePath);
            return this.volume.lstatSync(fullPath).isDirectory();
        });
    }
    deleteFileRecursive(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fullPath = this.getAbsolutePath(filePath);
            if (yield this.isDirectory(filePath)) {
                const subFiles = yield this.listFiles(filePath);
                for (const subFile of subFiles) {
                    yield this.deleteFileRecursive(subFile);
                }
                this.volume.rmdirSync(fullPath);
            }
            else {
                this.volume.unlinkSync(fullPath);
            }
        });
    }
    getAbsolutePath(relativePath) {
        return '/' + relativePath.filePath;
    }
}
exports.StubMemoryFileSystem = StubMemoryFileSystem;
//# sourceMappingURL=StubMemoryFileSystem.js.map