"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher = exports.FileSystemWithReservedSpace = void 0;
const fileSystemHelper_1 = require("./fileSystemHelper");
const httpHeadFetchers_1 = require("./httpHeadFetchers");
class FileSystemWithReservedSpace {
    constructor(fileSystem, httpHeadFetcher, reservedPercentage) {
        this.fileSystem = fileSystem;
        this.httpHeadFetcher = httpHeadFetcher;
        this.reservedPercentage = reservedPercentage;
    }
    writeFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const requestedFileSystem = yield this.getStorageUnitByType(filePath.storageUnit.type);
            if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, filePath.filePath, Buffer.byteLength(contents, 'utf-8')))) {
                throw new Error('Not enough free space.');
            }
            return yield this.fileSystem.writeFile(filePath, contents);
        });
    }
    listStorageUnits() {
        return __awaiter(this, void 0, void 0, function* () {
            const storageUnits = yield this.fileSystem.listStorageUnits();
            return storageUnits.map((storageUnit) => {
                const reservedSpace = (0, fileSystemHelper_1.calculateReservedSpace)(storageUnit.capacity, this.reservedPercentage);
                // Don't go below zero, in case we're already limited by platform.
                storageUnit.usableSpace = Math.max(0, storageUnit.usableSpace - reservedSpace);
                return storageUnit;
            });
        });
    }
    onStorageUnitsChanged(listener) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.fileSystem.onStorageUnitsChanged(listener);
        });
    }
    listFiles(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.listFiles(directoryPath);
        });
    }
    getFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.getFile(filePath);
        });
    }
    readFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.readFile(filePath);
        });
    }
    appendFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const requestedFileSystem = yield this.getStorageUnitByType(filePath.storageUnit.type);
            if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, filePath.filePath, Buffer.byteLength(contents, 'utf-8')))) {
                throw new Error('Not enough free space.');
            }
            return yield this.fileSystem.appendFile(filePath, contents);
        });
    }
    exists(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.exists(filePath);
        });
    }
    downloadFile(filePath, sourceUri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            let contentLength;
            try {
                const response = yield this.httpHeadFetcher(sourceUri, headers);
                contentLength = response.headers['content-length'];
            }
            catch (error) {
                console.warn(`Failed to process HEAD request on downloaded file ${sourceUri}`);
            }
            if (contentLength === undefined) {
                // This can't be determined due to functionality of gzip/chunk-enconding. https://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html#sec4.4
                console.warn(`No Content-Length header during downloadFile ${sourceUri}. In that case, we can not determine the full functionality of checking free space.`);
                contentLength = 0;
            }
            const requestedFileSystem = yield this.getStorageUnitByType(filePath.storageUnit.type);
            if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, filePath.filePath, Number(contentLength)))) {
                throw new Error('Not enough free space.');
            }
            return yield this.fileSystem.downloadFile(filePath, sourceUri, headers);
        });
    }
    uploadFile(filePath, uri, formKey, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.uploadFile(filePath, uri, formKey, headers);
        });
    }
    deleteFile(filePath, recursive) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.deleteFile(filePath, recursive);
        });
    }
    copyFile(sourceFilePath, destinationFilePath, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const file = yield this.fileSystem.getFile(sourceFilePath);
            if (file !== null) {
                const requestedFileSystem = yield this.getStorageUnitByType(destinationFilePath.storageUnit.type);
                if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, destinationFilePath.filePath, file === null || file === void 0 ? void 0 : file.sizeBytes))) {
                    throw new Error('Not enough free space.');
                }
            }
            return yield this.fileSystem.copyFile(sourceFilePath, destinationFilePath, options);
        });
    }
    moveFile(sourceFilePath, destinationFilePath, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const file = yield this.fileSystem.getFile(sourceFilePath);
            if (file !== null && sourceFilePath.storageUnit.type !== destinationFilePath.storageUnit.type) {
                const requestedFileSystem = yield this.getStorageUnitByType(destinationFilePath.storageUnit.type);
                if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, destinationFilePath.filePath, file.sizeBytes))) {
                    throw new Error('Not enough free space.');
                }
            }
            return yield this.fileSystem.moveFile(sourceFilePath, destinationFilePath, options);
        });
    }
    link(sourceFilePath, destinationFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.link(sourceFilePath, destinationFilePath);
        });
    }
    getFileChecksum(filePath, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.getFileChecksum(filePath, hashType);
        });
    }
    extractFile(archiveFilePath, destinationDirectoryPath, method) {
        return __awaiter(this, void 0, void 0, function* () {
            const { uncompressedSize } = yield this.fileSystem.getArchiveInfo(archiveFilePath);
            const requestedFileSystem = yield this.getStorageUnitByType(destinationDirectoryPath.storageUnit.type);
            if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, destinationDirectoryPath.filePath, uncompressedSize))) {
                throw new Error('Not enough free space.');
            }
            return yield this.fileSystem.extractFile(archiveFilePath, destinationDirectoryPath, method);
        });
    }
    createArchive(archiveFilePath, archiveEntries) {
        return __awaiter(this, void 0, void 0, function* () {
            const requestedFileSystem = yield this.getStorageUnitByType(archiveFilePath.storageUnit.type);
            const expectedArchiveSize = yield new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
                var _a;
                try {
                    let finalArchiveSize = 0;
                    for (const entry of archiveEntries) {
                        const file = yield this.fileSystem.getFile(entry);
                        if (file !== null) {
                            finalArchiveSize += (_a = file.sizeBytes) !== null && _a !== void 0 ? _a : 0;
                        }
                    }
                    resolve(finalArchiveSize);
                }
                catch (error) {
                    reject(error);
                }
            }));
            if (!(yield (0, fileSystemHelper_1.validateFreeSpace)(requestedFileSystem, archiveFilePath.filePath, expectedArchiveSize))) {
                throw new Error('Not enough free space.');
            }
            return yield this.fileSystem.createArchive(archiveFilePath, archiveEntries);
        });
    }
    createDirectory(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.createDirectory(directoryPath);
        });
    }
    isDirectory(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.isDirectory(filePath);
        });
    }
    getArchiveInfo(archiveFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.getArchiveInfo(archiveFilePath);
        });
    }
    wipeout() {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.fileSystem.wipeout();
        });
    }
    /** @deprecated Test only. */
    getExtendedFileSystem() {
        return this.fileSystem;
    }
    getStorageUnitByType(type) {
        return __awaiter(this, void 0, void 0, function* () {
            const storageUnits = yield this.listStorageUnits();
            const foundStorageUnit = storageUnits.find((storageUnit) => storageUnit.type === type);
            if (foundStorageUnit === undefined) {
                throw new Error(`StorageUnit for type ${type} not found`);
            }
            return foundStorageUnit;
        });
    }
}
exports.FileSystemWithReservedSpace = FileSystemWithReservedSpace;
function createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher(fileSystem, window, reservedPercentage) {
    const httpHeadFetcher = (0, httpHeadFetchers_1.createWindowHttpHeadFetcher)(window);
    return new FileSystemWithReservedSpace(fileSystem, httpHeadFetcher, reservedPercentage);
}
exports.createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher = createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher;
//# sourceMappingURL=FileSystemWithReservedSpace.js.map