// @ts-check
/* eslint-disable no-var */
/* 
Include console.js to any application to display a semi-transparent overlay
with contents of the web console. This is useful for environments where the
console is not available. This code should work in almost any browser and does
not need to be transpiled

Example use:
<script type="text/javascript" language="javascript" src="js/console.js"></script>
*/
'use strict';

var errorsMap = {};

// Polyfill for Error constructor with stack for older browsers
var ErrorOrig = Error;
var ErrorWithStack = function (message) {
	this.name = 'Error';
	if (!(this instanceof ErrorWithStack)) {
		return new ErrorWithStack(message);
	}
	var stack;
	try {
		throw new ErrorOrig();
	} catch (error) {
		stack = error.stack;
	}
	this.constructor = ErrorWithStack;
	this.message = message;
	this.stack = stack;
	errorsMap[message] = this;
};
ErrorWithStack.prototype = ErrorOrig.prototype;
window.Error = ErrorWithStack;

// Polyfill ErrorEvent.error of addEventListener('error') for older browsers
window.addEventListener('error', function (event) {
	if (event.error === undefined) {
		// Not standardized. ErrorEvent.message can be prefixed with 'Error: ' or 'Uncaught Error: ' based on version of chrome
		var errorMessage = Object.keys(errorsMap).find(function (key) {
			return event.message.indexOf(key) !== -1;
		});
		if (errorMessage) {
			var error = errorsMap[errorMessage];
			delete errorsMap[errorMessage];
			// @ts-ignore error is not defined in ErrorEvent for old browsers so it's not read-only
			event.error = error;
		}
	}
});

var DEFAULT_DEPTH = 3;

// Console settings
/** @type {ConsoleMethod[]} */
var patchedMethods = ['error', 'warn', 'info', 'log'];

/** @type {Record<string, string>} */
var colors = {
	error: 'lightcoral',
	warn: 'gold',
};

// Console settings end

/** @type {[string, unknown][] | null} */
var preMountQueue = [];

/** @type {HTMLTableElement} */
var log;

/**
 * @param {string} level
 * @param {unknown} content
 */
function reportToScreen(level, content) {
	if (preMountQueue === null) {
		var color = colors[level];
		if (!color) {
			color = 'white';
		}

		addLine(['[' + level + ']:', formatAny(content)], color);
	} else {
		preMountQueue.push([level, content]);
	}
}

/**
 * @param {(string | HTMLDivElement)[]} content
 * @param {string} color
 */
function addLine(content, color) {
	var row = document.createElement('tr');

	row.style.color = color;

	content.forEach(function (itm) {
		var td = document.createElement('td');
		td.style.verticalAlign = 'top';

		var pre = document.createElement('pre');
		pre.style.margin = '0';

		pre.appendChild(typeof itm === 'string' ? document.createTextNode(itm + '\n') : itm);
		td.appendChild(pre);
		row.appendChild(td);
	});

	var firstChild = log.childNodes[0];

	log.insertBefore(row, firstChild);
}

/**
 * @param {unknown} content
 * @param {number} [depth]
 * @returns {HTMLDivElement}
 */
function formatAny(content, depth) {
	depth = depth === undefined ? DEFAULT_DEPTH : depth;

	if (depth <= 0) {
		return indentedBlock(['{...}']);
	}

	/** @type {(string | HTMLDivElement)[]} */
	var lines = [];

	if (isInlineable(content)) {
		return indentedBlock([JSON.stringify(content)]);
	}

	if (Array.isArray(content)) {
		lines.push('[');
		content.forEach(function (itm) {
			lines.push(formatAny(itm));
		});
		lines.push(']');
		return indentedBlock(lines);
	}

	if (content instanceof Error) {
		lines.push('Error {');

		lines.push(indentedBlock(['name: "' + content.name + '"']));
		lines.push(indentedBlock(['message: "' + content.message + '"']));
		lines.push(indentedBlock(['stack: "' + String(content.stack) + '"']));
		lines.push('}');
		return indentedBlock(lines);
	}

	if (content instanceof ErrorEvent) {
		lines.push('ErrorEvent {');

		lines.push(indentedBlock(['message: "' + content.message + '"']));
		lines.push(indentedBlock(['filename: "' + content.filename.substr(content.filename.length - 50) + '"']));
		lines.push(indentedBlock(['lineno:colno: ' + content.lineno + ':' + content.colno]));
		lines.push(indentedBlock(['error: ', formatAny(content.error)]));
		lines.push('}');
		return indentedBlock(lines);
	}

	if (typeof content === 'bigint') {
		return indentedBlock([content + 'n']);
	}

	if (typeof content === 'symbol' || content === undefined) {
		return indentedBlock([String(content)]);
	}

	if (typeof content === 'function') {
		var name = content.name;
		if (!name) {
			name = 'function';
		}
		return indentedBlock([name + '()']);
	}

	if (content && typeof content === 'object') {
		lines.push('{');
		Object.keys(content).forEach(function (key) {
			lines.push(indentedBlock([key + ': ', formatAny(content[key], depth - 1)]));
		});
		lines.push('}');
		return indentedBlock(lines);
	}

	try {
		return indentedBlock([JSON.stringify(content)]);
	} catch (e) {
		return indentedBlock(['{ Unable to display value ' + String(content) + ' }']);
	}
}

/**
 * @param {(string | HTMLDivElement)[]} lines
 */
function indentedBlock(lines) {
	var block = document.createElement('div');
	block.style.paddingLeft = '1em';
	block.style.whiteSpace = 'pre';

	lines.forEach(function (itm) {
		block.appendChild(typeof itm === 'string' ? document.createTextNode(itm + '\n') : itm);
	});
	return block;
}

/**
 * All console keys which values are callable
 * @typedef {{ [K in keyof Console]: Console[K] extends (...x: any[]) => any ? K : never}[keyof Console]} ConsoleMethod
 */

/**
 * @param {ConsoleMethod} level
 */
function patch(level) {
	var original = window.console[level];

	var func = function () {
		var args = Array.prototype.slice.call(arguments);
		reportToScreen(level, args);
		original.apply(console, args);
	};
	window.console[level] = func.bind(window.console);
}

/**
 * @param {unknown} content
 * @param {number} [depth]
 * @returns {boolean}
 */
function isInlineable(content, depth) {
	depth = depth === undefined ? DEFAULT_DEPTH : depth;
	if (depth <= 0) {
		return false;
	}

	if (Array.isArray(content)) {
		return content.every(function (c) {
			return isInlineable(c, depth - 1);
		});
	}

	if (isPlainObject(content)) {
		return Object.keys(content).every(function (k) {
			return isInlineable(/** @type {any} */ content[k], depth - 1);
		});
	}

	return typeof content === 'boolean' || typeof content === 'number' || typeof content === 'string' || content === null;
}

/**
 * @param {unknown} value
 * @returns {value is object}
 */
function isPlainObject(value) {
	if (typeof value !== 'object' || value === null) {
		return false;
	}

	var proto = Object.getPrototypeOf(value);
	return proto !== null && Object.getPrototypeOf(proto) === null;
}

patchedMethods.forEach(function (m) {
	patch(m);
});

function init() {
	if (!preMountQueue) {
		return;
	}

	log = document.createElement('table');
	document.body.appendChild(log);

	log.style.position = 'absolute';
	log.style.top = '0';
	log.style.left = '0';
	log.style.right = '0';
	log.style.bottom = '0';
	log.style.zIndex = '99999';
	log.style.textShadow = '-1px -1px 0 #000, 1px -1px 0 #000, -1px 1px 0 #000, 1px 1px 0 #000';
	log.style.color = 'white';
	log.style.whiteSpace = 'pre';
	log.style.fontFamily = "'courier new',monospace";

	var q = preMountQueue;
	preMountQueue = null;

	q.forEach(function (entry) {
		reportToScreen(entry[0], entry[1]);
	});
}

var documentReady = document.readyState === 'complete' || document.readyState === 'interactive';

if (documentReady) {
	init();
} else {
	window.addEventListener('DOMContentLoaded', function () {
		init();
	});
}

window.addEventListener('error', function () {
	var args = Array.prototype.slice.call(arguments);
	reportToScreen('onerror', args);
});

reportToScreen('ready', 'On screen console has loaded');
