"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SyncEngine = void 0;
const events_1 = require("events");
const syncMessages_1 = require("./syncMessages");
const Validate_1 = require("../Validate/Validate");
var SyncEngine;
(function (SyncEngine) {
    SyncEngine["SyncServer"] = "sync-server";
    SyncEngine["P2PLocal"] = "p2p-local";
    /** @deprecated use P2PLocal */
    SyncEngine["Udp"] = "udp";
})(SyncEngine = exports.SyncEngine || (exports.SyncEngine = {}));
var SyncEvent;
(function (SyncEvent) {
    SyncEvent["Status"] = "status";
    SyncEvent["BroadcastedValue"] = "broadcasted_value";
    SyncEvent["Closed"] = "closed";
})(SyncEvent || (SyncEvent = {}));
class Sync {
    constructor(messagePrefix, postMessage) {
        this.messagePrefix = messagePrefix;
        this.postMessage = postMessage;
        this.eventEmitter = new events_1.EventEmitter();
    }
    connect(options) {
        return __awaiter(this, void 0, void 0, function* () {
            // TODO switch default to UDP once it's mature enough
            const defaultOptions = { engine: SyncEngine.SyncServer };
            let sanitizedOptions;
            if (typeof options === 'string') {
                // support legacy connect(uri) method, which uses the sync-server engine
                sanitizedOptions = {
                    engine: SyncEngine.SyncServer,
                    uri: options,
                };
            }
            else if (options === undefined) {
                sanitizedOptions = defaultOptions;
            }
            else {
                sanitizedOptions = options;
            }
            if (sanitizedOptions.engine === SyncEngine.SyncServer) {
                Validate_1.default({ uri: sanitizedOptions.uri }).uri();
            }
            yield this.postMessage({
                type: this.getMessage('connect'),
                engine: sanitizedOptions.engine,
                serverUri: sanitizedOptions.uri,
            });
        });
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.postMessage({
                type: this.getMessage('close'),
            });
        });
    }
    /** @deprecated use joinGroup */
    init(groupName = Sync.DEFAULT_GROUP_NAME, deviceIdentification) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.joinGroup({ groupName, deviceIdentification });
        });
    }
    joinGroup({ groupName, deviceIdentification }) {
        return __awaiter(this, void 0, void 0, function* () {
            groupName = groupName !== null && groupName !== void 0 ? groupName : Sync.DEFAULT_GROUP_NAME;
            Validate_1.default({ groupName }).required().string();
            Validate_1.default({ deviceIdentification }).string();
            yield this.postMessage({
                type: this.getMessage("init"),
                groupName,
                deviceIdentification,
            });
        });
    }
    leaveGroup(groupName = Sync.DEFAULT_GROUP_NAME) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ groupName }).required().string();
            yield this.postMessage({
                type: this.getMessage("leave_group"),
                groupName,
            });
        });
    }
    wait(data, groupName = Sync.DEFAULT_GROUP_NAME, timeout) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ groupName }).required().string();
            Validate_1.default({ timeout }).number();
            const response = yield this.postMessage({
                type: this.getMessage('wait'),
                groupName,
                data,
                timeout,
            });
            return response.responseData;
        });
    }
    cancelWait(groupName = Sync.DEFAULT_GROUP_NAME) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ groupName }).required().string();
            yield this.postMessage({
                type: this.getMessage('cancel_wait'),
                groupName,
            });
        });
    }
    /** @deprecated use broadcastValue */
    setValue(key, value, groupName = Sync.DEFAULT_GROUP_NAME) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.broadcastValue({ key, value, groupName });
        });
    }
    broadcastValue({ groupName, key, value }) {
        return __awaiter(this, void 0, void 0, function* () {
            groupName = groupName !== null && groupName !== void 0 ? groupName : Sync.DEFAULT_GROUP_NAME;
            Validate_1.default({ key }).required().string();
            Validate_1.default({ groupName }).required().string();
            yield this.postMessage({
                type: this.getMessage('request_set_value'),
                groupName,
                key,
                value,
            });
        });
    }
    onValue(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.addListener(SyncEvent.BroadcastedValue, (event) => {
            listener(event.key, event.value, event.groupName === Sync.DEFAULT_GROUP_NAME ? undefined : event.groupName);
        });
    }
    onStatus(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.addListener(SyncEvent.Status, (event) => {
            listener({ connectedPeers: event.connectedPeers, groupName: event.groupName === Sync.DEFAULT_GROUP_NAME ? undefined : event.groupName });
        });
    }
    onClosed(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.addListener(SyncEvent.Closed, listener);
    }
    handleMessageData(data) {
        switch (data.type) {
            case this.getMessage(syncMessages_1.SyncMessageType.BroadcastedValue):
                const broadcastedValueMessage = data;
                const valueEvent = {
                    groupName: broadcastedValueMessage.groupName,
                    key: broadcastedValueMessage.key,
                    value: broadcastedValueMessage.value,
                };
                this.eventEmitter.emit(SyncEvent.BroadcastedValue, valueEvent);
                break;
            case this.getMessage(syncMessages_1.SyncMessageType.Status):
                const statusMessage = data;
                const statusEvent = {
                    connectedPeers: statusMessage.connectedPeers,
                    groupName: statusMessage.groupName,
                };
                this.eventEmitter.emit(SyncEvent.Status, statusEvent);
                break;
            case this.getMessage(syncMessages_1.SyncMessageType.Closed):
                const closedMessage = data;
                const closedEvent = closedMessage.error ? new Error(closedMessage.error) : undefined;
                this.eventEmitter.emit(SyncEvent.Closed, closedEvent);
            default:
        }
    }
    removeEventListeners() {
        this.eventEmitter.removeAllListeners();
    }
    getMessage(name) {
        return this.messagePrefix + '.' + Sync.MESSAGE_PREFIX + '.' + name;
    }
}
exports.default = Sync;
Sync.MESSAGE_PREFIX = 'sync';
Sync.DEFAULT_GROUP_NAME = '____default____';
//# sourceMappingURL=Sync.js.map