"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const StreamProtocol_1 = require("./StreamProtocol");
const AppletStreamError_1 = require("../Error/AppletStreamError");
const ErrorCodes_1 = require("../Error/ErrorCodes");
const Validate_1 = require("../Validate/Validate");
const coordinationsHelper_1 = require("../Dimensions/coordinationsHelper");
class Stream {
    constructor(messagePrefix, postMessage) {
        this.messagePrefix = messagePrefix;
        this.postMessage = postMessage;
        this.eventEmitter = new events_1.EventEmitter();
    }
    prepare(uri, x, y, width, height, protocol, options = {}) {
        this.checkParamsValidity(uri, x, y, width, height);
        this.createAndEmitEvent('prepare', {
            type: 'prepare',
            srcArguments: {
                uri, x, y, width, height, protocol, options,
            },
        });
        return this.postMessage({
            type: this.getMessage('prepare'),
            uri, x, y, width, height, protocol, options,
        });
    }
    play(uri, x, y, width, height, protocol, options = {}) {
        this.checkParamsValidity(uri, x, y, width, height);
        ({ x, y, width, height } = coordinationsHelper_1.sanitizeCoordinations({ x, y, width, height }));
        let uppercaseProtocol;
        if (typeof protocol !== 'undefined') {
            uppercaseProtocol = protocol.toUpperCase(); // backwards compatibility, old format was lowercase
            if (Object.keys(StreamProtocol_1.default).indexOf(uppercaseProtocol) < 0) {
                throw new AppletStreamError_1.default({
                    kind: 'appletStreamError',
                    message: 'Invalid stream protocol type: ' + protocol,
                    code: ErrorCodes_1.default.STREAM_INVALID_PROTOCOL_TYPE,
                });
            }
        }
        this.createAndEmitEvent('play', {
            type: 'play',
            srcArguments: {
                uri, x, y, width, height, protocol, options,
            },
        });
        return this.postMessage({
            type: this.getMessage('play'),
            uri,
            x,
            y,
            width,
            height,
            protocol: uppercaseProtocol,
            options,
        });
    }
    stop(uri, x, y, width, height) {
        this.checkParamsValidity(uri, x, y, width, height);
        ({ x, y, width, height } = coordinationsHelper_1.sanitizeCoordinations({ x, y, width, height }));
        this.createAndEmitEvent('stop', {
            type: 'stop',
            srcArguments: {
                uri, x, y, width, height,
            },
        });
        return this.postMessage({
            type: this.getMessage('stop'),
            uri,
            x,
            y,
            width,
            height,
        });
    }
    onError(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.on('error', listener);
    }
    onConnected(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.on('connected', listener);
    }
    onDisconnected(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.on('disconnected', listener);
    }
    onPrepare(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.on('prepare', listener);
    }
    onPlay(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.on('play', listener);
    }
    onStop(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.on('stop', listener);
    }
    handleMessageData(data) {
        switch (data.type) {
            case this.getMessage('error'):
                this.createAndEmitEvent('error', {
                    type: 'error',
                    srcArguments: {
                        uri: data.uri,
                        x: data.x,
                        y: data.y,
                        width: data.width,
                        height: data.height,
                        protocol: data.protocol,
                        options: data.options,
                    },
                    errorMessage: data.errorMessage,
                });
                break;
            case this.getMessage('connected'):
                this.createAndEmitEvent('connected', {
                    type: 'connected',
                    srcArguments: {
                        uri: data.uri,
                        x: data.x,
                        y: data.y,
                        width: data.width,
                        height: data.height,
                        protocol: data.protocol,
                        options: data.options,
                    },
                });
                break;
            case this.getMessage('disconnected'):
                this.createAndEmitEvent('disconnected', {
                    type: 'disconnected',
                    srcArguments: {
                        uri: data.uri,
                        x: data.x,
                        y: data.y,
                        width: data.width,
                        height: data.height,
                        protocol: data.protocol,
                        options: data.options,
                    },
                });
                break;
            default:
        }
    }
    removeEventListeners() {
        this.eventEmitter.removeAllListeners();
    }
    createAndEmitEvent(type, event) {
        this.eventEmitter.emit(type, event);
    }
    getMessage(name) {
        return this.messagePrefix + '.' + Stream.MESSAGE_PREFIX + '.' + name;
    }
    checkParamsValidity(uri, x, y, width, height) {
        Validate_1.default({ uri }).required().uri();
        Validate_1.default({ x }).required().number();
        Validate_1.default({ y }).required().number();
        Validate_1.default({ width }).required().number();
        Validate_1.default({ height }).required().number();
    }
}
exports.default = Stream;
Stream.MESSAGE_PREFIX = 'stream';
//# sourceMappingURL=Stream.js.map