---
title:  Serial
author: Michal Artazov
date:   25.9.2020
type:   js-api
tags:
- applet
- applet_api
- api
- serial
- js_api
description: "[Content JS API] Some platforms support connecting hardware via serial port, either directly via UART or via USB. This can be leveraged to integrate a wide variety of hardware into your digital signage application."
---

# Serial

Some platforms support connecting hardware via serial port, either directly via UART or via USB.
This can be leveraged to integrate a wide variety of hardware into your digital signage application.

Such hardware could be, for example, printers, payment terminals or sensors.

::: alert alert--warning
* Before using this API, ensure that the display supports serial via `sos.display.supports("SERIAL")`.
* More info [**HERE**](/api/js/content/latest/js-display#supports).
* Samsung Kiosk serial connection only works over serial ports, not over USB ports
:::

## All methods

::: table-responsive
| Method             | Description            | Supported since |
| ------------------ | ---------------------- | :-------------: |
| `openSerialPort()` | Open a new serial port | 4.4.0           |
:::

## openSerialPort()

Creates a new instance of serial port. You can then use that instance to handle the communication.

::: table-responsive
| Param     | Type   | Required                   | Description                                        |
| --------- | ------ | :------------------------: | -------------------------------------------------- |
| `options` | Object | <div class="red">Yes</div> | Object containing configuration of the serial port |
:::

### Options

Here is the options object defined as Typescript interface:

```typescript
interface ISerialPortOptions {
    device?: string;
    baudRate: number;
    parity?: 'none' | 'even' | 'mark' | 'odd' | 'space';
    databits?: number;
    stopbits?: number;
}
```

### Javascript example
```javascript
// Open default serial port based platform
const serialPort = await sos.hardware.openSerialPort({
    baudRate: 115200,
});

// Open specific serial port
const serialPort = await sos.hardware.openSerialPort({
    device: '/dev/ttyUSB0',
    baudRate: 115200,
});
```

::: alert alert--info
Device address is different per platform (at least for now). Kindly find the respective device address:

| Device type | Default value | Other values for `device` |
| ----------- | ---------- | ----------------- |
| RaspberryPi / Linux | `/dev/ttyUSB0` | `/dev/ttyUSB0` |
| Windows | `COM3` | `COM3` (typically, but check your Win machine if this does not work) |
| Samsung Kiosk | `PORT1` |`PORT1`, `PORT2`, `PORT3` |
| Android | `/dev/ttyusb0` | `/dev/ttyusb0` |
| BrightSign | `0` | `0`, `1`, `USB:A/0` |
:::

## Serial port methods

Once you create a serial port instance using `openSerialPort()` method listed above,
you can call following methods on it.

::: table-responsive
| Method     | Description                                                 | Supported since |
| ---------- | ----------------------------------------------------------- | :-------------: |
| `onData()` | Call a listener callback anytime the opened serial port receiving data. | 4.4.0           |
| `write()`  | Write data to the serial port                               | 4.4.0           |
| `close()`  | Close serial port                                           | 4.4.0           |
:::

## onData()

Call a listener callback anytime the opened serial port receiving data.
The listener will stop if the serial port is closed.

Since data may be either text or binary, it's emitted as [Uint8Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Uint8Array)
and should be processed further according to the requirements of your application.

### Javascript example
In case of binary data:
```javascript
// serial port instance previously created via sos.hardware.openSerialPort()
serialPort.onData((data) => {
    // data is array of numbers so can be processed as binary
});
```

In case of text data:
```javascript
// serial port instance previously created via sos.hardware.openSerialPort()
serialPort.onData((data) => {
    const dataString = [...data].map((char) => String.fromCharCode(char)).join('');
});
```

## write()

Write data to the serial port. The data can be a string of hexadecimal digits, array of numbers or [Uint8Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Uint8Array). And converted to hexadecimals

### Javascript example
```javascript
// serial port instance previously created via sos.hardware.openSerialPort()
await serialPort.write('68656c6c6f'); // hexadecimal string
await serialPort.write([ 10, 20, 30, 40 ]); // array of numbers
await serialPort.write(Uint8Array.from([ 10, 20, 30, 40 ])); // Uint8Array
```

## close()

Close serial port and prevent any more communication

### Javascript example
```javascript
// serial port instance previously created via sos.hardware.openSerialPort()
await serialPort.close();
```

## Errors

Following errors may occur when working with the serial ports. However they don't reflect on actual physical connectivity of the ports due to Samsung b2bapi does not register these events.


::: table-responsive
| Code  | Type                  | Message                                   |
| ----- | --------------------- | ----------------------------------------- |
| 50602 | InternalHardwareError | Failed to open serial port                |
| 50603 | InternalHardwareError | Closing serial port that isn't open       |
| 50603 | InternalHardwareError | Failed to close serial port               |
| 50604 | InternalHardwareError | Writing to a serial port that isn't open |
| 50604 | InternalHardwareError | Failed to write to serial port            |
:::
