---
title:  File System
author: Michael Zabka
date:   26.11.2018
type:   js-api
tags:
- applet
- applet_api
- api
- js_api
description: "[Content JS API] File System API allows **low level manipulation with files and directories** in the device internal storage. File System API works with both in **internal and external storages** (USB flash drives, etc.)."
---

# File System
File System API allows **low level manipulation with files and directories** in the device internal storage.
File System API works with both in **internal and external storages** (USB flash drives, etc.).

::: alert alert--warning
File System directory structure is **PERSISTENT** and is **NOT** automatically deleted through `Applet Reload` power action! Applet Reload only deletes `/data` directory which is reserved for simplified [Offline Cache API](/api/js/content/js-offline-cache-media-files). Use `deleteFile()` to clear the device storage file system.
:::

<div class="row d-flex align-content-stretch force-padding-10">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" target="_blank" href="https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/file-system">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fab fa-github svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Complete example of the File System API on our Github</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

<div class="row d-flex align-content-stretch force-padding-20">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" href="/api/js/content/js-offline-cache-media-files">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fas fa-medal svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Offline Cache for media files</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

## All methods

::: table-responsive
| Methods  | Description | Supported since |
| -------- | ----------- | :---------------: |
| `listStorageUnits()` | Return list of storage units /internal, external/ from the device | 2.1.0 |
| `listFiles()` | Return list of files of existing directory | 2.1.0 |
| `getFile()` | Returns local uri of existing file | 2.1.0 |
| `exists()` | Returns true of existing directory or file | 2.1.0 |
| `downloadFile()` | Download file to path within existing directory or Overrides existing file in path when successfully downloaded | 2.1.0 |
| `deleteFile()` | Delete file from path within existing directory | 2.1.0 |
| `moveFile()` | Move file to destination path or Move directory to destination path | 2.1.0 |
| `getFileChecksum()` | Returns checksum of existing file in path | 2.1.0 |
| `extractFile()` | Extract ZIPed all files (recursively) to destination path and override existing files or merge existing directories | 2.1.0 |
| `createArchive()` | Creates archive in selected path | 5.12.0 |
| `createDirectory()` | Creates directory in path | 2.1.0  |
| `isDirectory()` | Returns true if existing path is directory | 2.1.0 |
| `copyFile()` | Copy file or directory to a new location | 3.3.0 |
| `writeFile()` | Write string to a file | 3.2.0 |
| `appendFile()` | Write string to a file | 4.13.0 |
| `readFile()` | Read a text file | 3.3.0 |
:::

## listStorageUnits()

Return list of storage units /internal, external/ from the device. Capacity values are in bytes.

::: alert alert--info
This is a mandatory method that is required for all the other File System APIs. The other APIs require a `storageUnit` object that is retrieved from this method to manipulate with files on a correct storage location (internal/external).
:::

::: alert alert--warning
`storageUnit` is a dynamic object! It has to be always generated and retrieved by this JS API, as the values in `type` differ platform by platform. **Never** generate the object manually. `{"type":"internal"}` is for demonstration only.
:::


### Javascript Example

```javascript
// Storage units are equivalent to disk volumes (C:, D: etc on Windows; /mnt/disc1, /mnt/disc2 on Unix)
const storageUnits = await sos.fileSystem.listStorageUnits();

// Every platform has at least one not removable storage unit (internal storage unit)
const internalStorageUnit = storageUnits.find((storageUnit) => !storageUnit.removable);

console.log(storageUnits);
```

### Returns

```json
[
   {
      "type":"internal",
      "capacity":4124856000, // in bytes
      "freeSpace":3764700000, // in bytes
      "usableSpace":3764700000, // in bytes
      "removable":false
   },
   {
      "type":"external",
      "capacity":2224856000, // in bytes
      "freeSpace":764700000, // in bytes
      "usableSpace":764700000, // in bytes
      "removable":true
   }
]
```

## listFiles()

List files and folders located in the internal/external storage.
* SUCCESS: Return list of files of existing directory
* FAIL: When listing not existing directory
* FAIL: When listing existing file (not a directory)

### Javascript Example

```javascript
// Storage units are equivalent to disk volumes (C:, D: etc on Windows; /mnt/disc1, /mnt/disc2 on Unix)
const storageUnits = await sos.fileSystem.listStorageUnits();

// Every platform has at least one not removable storage unit (internal storage unit)
const internalStorageUnit = storageUnits.find((storageUnit) => !storageUnit.removable);

// Empty string '' refers to root directory. Here is root directory of internal storage
const rootDirectoryFilePaths = await sos.fileSystem.listFiles({ filePath: '', storageUnit: internalStorageUnit });

console.log(rootDirectoryFilePaths);

// get files from directory 'test-dir'
const testDirDirectoryFilePaths = await sos.fileSystem.listFiles({ filePath: 'test-dir', storageUnit: internalStorageUnit });
console.log(testDirDirectoryFilePaths);
```

### Returns

```json
// there are 2 directories in the internal storage
[
   {
      "storageUnit":{
         "type":"internal",
         "capacity":4124856000,
         "freeSpace":3336336000,
         "usableSpace":3336336000,
         "removable":false
      },
      "filePath":"data"
   },
   {
      "storageUnit":{
         "type":"internal",
         "capacity":4124856000,
         "freeSpace":3336336000,
         "usableSpace":3336336000,
         "removable":false
      },
      "filePath":"test-dir"
   }
]

// there are 3 files in the 'test-dir' directory
[
   {
      "storageUnit":{
         "type":"internal",
         "capacity":4124856000,
         "freeSpace":3764696000,
         "usableSpace":3764696000,
         "removable":false
      },
      "filePath":"test-dir/downloaded-file.png"
   },
   {
      "storageUnit":{
         "type":"internal",
         "capacity":4124856000,
         "freeSpace":3764696000,
         "usableSpace":3764696000,
         "removable":false
      },
      "filePath":"test-dir/log.1.txt"
   },
   {
      "storageUnit":{
         "type":"internal",
         "capacity":4124856000,
         "freeSpace":3764696000,
         "usableSpace":3764696000,
         "removable":false
      },
      "filePath":"test-dir/log.1.txt.backup"
   }
]
```

## getFile()

Return a file from an internal storage
* OK: Returns local uri of existing file
* OK: Returns NULL of not existing file

### Parameters

| Param | Type | Required | Description |
| -------------- | ------ | :-------: | ----- |
| `filePath`    | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example

```javascript
// always pass storageUnit to tell the JS API where to look
// get properties of file (at least localUri) or null when not exist
const fileProperties = await sos.fileSystem.getFile({ storageUnit: internalStorageUnit, filePath: 'test-dir/downloaded-file.png' });

// Complete file details
console.log(fileProperties);

// To access file from HTML applet, use localUri
console.log(fileProperties.localUri);
```

### Returns

::: alert alert--warning
Return statement is a dynamic object! It has to be always generated and retrieved by this JS API, as the values in `localUri` differ platform by platform. **Never** generate the object manually. `{"localUri":"file://internal/test-dir/downloaded-file.png"}` is for demonstration only.
:::

```json
{
   "localUri": "file://internal/test-dir/downloaded-file.png",
   // Other optional properties when supported
   "createdAt": "2019-05-03T12:00:00.000Z",
   "lastModifiedAt": "2019-05-03T12:00:00.000Z",
   "sizeBytes": 2048,
   "mimeType": "image/png"
}
```

## exists()

* OK: Returns true of existing directory or file
* OK: Returns false of not existing directory nor file

### Parameters

|      Param     | Type | Required | Description |           					                                                    |
| -------------- | ----- | :------: | ------ |
| `filePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example

```javascript
// always pass storageUnit to tell the JS API where to look
// check if there is such a file
const fileExists = await sos.fileSystem.exists({ storageUnit: internalStorageUnit, filePath: 'test-dir/downloaded-file.png' });

console.log(fileExists);
```

### Returns

Boolean value

## downloadFile()

Download file to path within existing directory or **overrides existing file** in a path when successfully downloaded

* OK: Download file to path within existing directory
* OK: Overrides existing file in path when sucessfully downloaded
* FAIL: When containing (parent) directory of destination path doesn't exist
* FAIL: When destination path is directory

### Parameters

|      Param      | Type  | Required | Description |             					                                                    |
| -------------- | ----- | :----: | ---- |
| `filePath` | FilePath  | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `uri` | String | <div class="red">Yes</div> | Url address to retrieve the file	from network			|
| `headers`  | Object | <div class="yellow">No</div> | Key, value pairs of HTTP headers to send along with the request. Used when the target file is protected by a password or if any 

::: alert alert--danger
For every download request, our app makes HEAD request for `content-length` header on that downloaded file.
:::

::: alert alert--warning
`headers` has to be a JSON object. If you are passing the value, make sure you use `JSON.parse()`.
:::

### Javascript Example

```javascript
// You can download file to specific existing directory
await sos.fileSystem.downloadFile(
    { storageUnit: internalStorageUnit, filePath: 'test-dir/downloaded-file.png' },
    'https://2.signageos.io/assets/android-benq-amy_bbd9afbc0655ceb6da790a80fbd90290.png',
    { 'Authentication-Token': 'MySecretToken' }, // Optionally, you can use headers for download file
);
```

#### Encoding

All downloads respect a standard of `Encoding` with optional compression of files. See [Mozilla standard Accept Encoding](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Accept-Encoding) and [Content Encoding](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Encoding).

Download file method is always sending optional following headers:
```
Accept-Encoding: gzip
Accept-Encoding: compress
Accept-Encoding: deflate
Accept-Encoding: br
Accept-Encoding: identity
Accept-Encoding: *
```

If the server understands the `Encoding` standard, it compresses files using `gzip` algorithm before the files are sent to the client. If so, the response will contain the following headers:
```
Content-Encoding: gzip
Content-Encoding: compress
Content-Encoding: deflate
Content-Encoding: br
```
So the data communication is compressed under the hood. The client will automatically decompress data before it's saved to a specified location path. So from JS API point of view, there is no requirement to decompress data by itself.

The standard is supported on all following platforms:

- WebOS 3+
- Tizen 2.4+
- Brightsign
- Raspberry Pi
- Windows

## deleteFile()

Delete a file or directory from the path.

* OK: (not recursive) Deletes an existing file or existing empty directory
* OK: (recursive) Deletes an existing file or existing (non-empty) directory recursively
* FAIL: (not recursive) When is deleting directory and is not empty
* FAIL: (both) When deleting path doesn't exist

### Parameters

| Param | Type | Required | Description |
| ---- | ----- | :------: | ----- |
| `filePath` | FilePath  | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `recursive`| boolean  | <div class="yellow">No</div> | `TRUE` for recursive delete of all content from a directory			|

### Javascript Example

```javascript
// Delete directory and all files inside
//// First check, if there is such a directory
if (await sos.fileSystem.exists({ storageUnit: internalStorageUnit, filePath: 'test-dir' })) {
    // Delete the directory and all it's content recursively
    await sos.fileSystem.deleteFile({ storageUnit: internalStorageUnit, filePath: 'test-dir' }, true);
}

// Delete file
//// First check, if there is such a file
if (await sos.fileSystem.exists({ storageUnit: internalStorageUnit, filePath: 'test-dir/downloaded-file.png' })) {
    // Delete the file
    await sos.fileSystem.deleteFile({ storageUnit: internalStorageUnit, filePath: 'test-dir/downloaded-file.png' }, false);
}
```

## moveFile()

Move file OR directory to destination path or Move directory to destination path.

* OK: Move file to destination path
* OK: Move directory to destination path
* FAIL: When destination path exists (is file or directory)
* FAIL: When containing (parent) directory of destination path doesn't exist
* FAIL: When source file (or directory) doesn't exist

### Parameters

| Param| Type | Required | Description |
| ------ | ------ | :-------: | ----- |
| `filePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example

```javascript
// Move file to a different directory
//// First check, if there is such a file and directory
if (await sos.fileSystem.exists({ storageUnit: internalStorageUnit, filePath: 'test-dir/log.txt' }) 
    && await sos.fileSystem.exists({ storageUnit: internalStorageUnit, filePath: 'new-dir' }) ) {
    
    await sos.fileSystem.moveFile(
		   { storageUnit: internalStorageUnit, filePath: 'test-dir/log.txt' },
         { storageUnit: internalStorageUnit, filePath: 'new-dir/log.txt' },
    );
}

```

## getFileChecksum()

Returns checksum of existing file in a path.
* OK: Returns checksum of existing file in path
* FAIL: When file path doesn't exist
* FAIL: When file path is directory

::: alert alert--info
Does not work on Samsung Tizen display - [read more here](https://docs.signageos.io/hc/en-us/articles/4416327809810)
:::

### Parameters

| Param | Type | Required | Description |
| ---- | ----- | :---------: | ------ |
| `filePath`    | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `hashFunction`   | string | <div class="red">Yes</div> | Available function is `md5` |

### Javascript Example

```javascript
const md5Checksum = await sos.fileSystem.getFileChecksum({ storageUnit: internalStorageUnit, filePath: 'test-dir/log.1.txt.backup' }, 'md5');
console.log(md5Checksum);
```

### Returns

```json
"b3c6930b9306b8e35a978d342cf5a01e" // string
```

## extractFile()

Extract ZIPed all files (recursively) to destination path and override existing files or merge existing directories.

* OK: Extract ZIPed all files (recursively) to destination path
* OK: Extract ZIPed all files (recursively) to destination path and override existing files or merge existing directories
* OK: Auto create destination directory path if doesn't exist
* FAIL: When extracting not existing ZIP archive file
* FAIL: When extracting try override existing directory with file or existing file with directory

::: alert alert--info
The directory/folder you are extracting your ZIP file into has to be created BEFORE you start extracting the ZIP.
:::

### Parameters

| Param | Type | Required | Description |
| ------ | ----- | -------- | ----- |
| `archiveFilePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `destinationFilePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `archiveMethod` | string | <div class="red">Yes</div> | Available method is `zip` |

### Javascript Example

```javascript
// Create directory 'test-dir' in a root directory
await sos.fileSystem.createDirectory({ storageUnit: internalStorageUnit, filePath: 'test-dir' });

// Extract ZIP file into the test-extracted directory
await sos.fileSystem.extractFile(
    { storageUnit: internalStorageUnit, filePath: 'test-dir/test.zip' },
    { storageUnit: internalStorageUnit, filePath: 'test-dir/test-extracted' },
    'zip',
);
```

## createArchive()

Creates a archive in a path with all files from `archiveEntries` array.

* OK: Creates directory in path
* FAIL: When creating directory over existing file or directory
* FAIL: When creating directory in not existing containing (parent) directory

::: alert alert--warning
- Never start OR end the `filePath` with a slash - `/`. It will cause error *50517*
- It is a good practice to check if file exists - `exists()` prior creating it
:::

::: alert alert--info
- This function is available only on Tizen devices.
- All files are added to the archive based on absolute path from root directory.
:::

### Parameters

| Param | Type | Required | Description |
| ------ | ----- | -------- | ----- |
| `archiveFilePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `archiveEntries` | FilePath Array | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and array of all files which will be archived into newly created archive |

```javascript
// Create archive 'test.zip' in a root directory with selected files
await sos.fileSystem.createArchive(
    { storageUnit: internalStorageUnit, filePath: 'test.zip' },
    [
        { storageUnit: internalStorageUnit, filePath: 'test-dir/log.txt' },
        { storageUnit: internalStorageUnit, filePath: 'test-dir/log.1.txt' },
    ],
);

// Example with listFiles() function if you want to archive all files from directory
const filesToArchive = await sos.fileSystem.listFiles({ storageUnit: internalStorageUnit, filePath: 'test-dir' });
await sos.fileSystem.createArchive(
    { storageUnit: internalStorageUnit, filePath: 'test.zip' },
    filesToArchive,
);
```

## createDirectory()

Creates a directory in a path.

* OK: Creates directory in path
* FAIL: When creating directory over existing file or directory
* FAIL: When creating directory in not existing containing (parent) directory

::: alert alert--warning
- Never start OR end the `filePath` with a slash - `/`. It will cause error *50512*
- It is a good practice to check if directory exists - `isDirectory()` prior creating it
:::

### Parameters

| Param | Type | Required | Description  |
| ----- | --------- | :-----: | --------- |
| `filePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example

```javascript
// Create directory 'test-dir' in a root directory
await sos.fileSystem.createDirectory({ storageUnit: internalStorageUnit, filePath: 'test-dir' });
```

```javascript
// Create multiple directories
const myDirs = [
 'dir1',
 'dir2',
 'dir1/dir3',
];
await Promise.all(myDirs.map(async (dirname) => {
 const fp = { storageUnit: internalStorageUnit, filePath: `uploads/${dirname}` };
 if (!await sos.fileSystem.exists(fp)) {
   await sos.fileSystem.createDirectory(fp);
 }
});
// All dirs created
```

## isDirectory()

Returns true if existing path is directory.

* OK: Returns true if existing path is directory
* OK: Returns false if existing path is file
* FAIL: When path doesn't exist

### Parameters

| Param | Type | Required | Description |
| ------ | --------------- | :-----------: | ------ | 
| `filePath` | FilePath  | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example

```javascript
await sos.fileSystem.isDirectory({ storageUnit: internalStorageUnit, filePath: 'test-dir' });
```

## copyFile()

Copy file OR directory to a new location.

### Parameters

| Param | Type | Required | Description |
| ------ | -------- | :----------: | ------ |
| `sourceFilePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `destinationFilePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example

```javascript
await sos.fileSystem.copyFile(
   { storageUnit: internalStorageUnit, filePath: 'test-dir/log.1.txt' },
   { storageUnit: internalStorageUnit, filePath: 'destination-dir/log.1.txt.backup' },
);
```

## writeFile()

Write into file directly:

- string support only
- override only
- no append

### Parameters

| Param | Type | Required | Description |
| ------ | -------- | :------: | ---- |
| `filePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `content` | String | <div class="red">Yes</div> | Parsed content as string |


### Javascript Example

```javascript
await sos.fileSystem.writeFile(
   { storageUnit: internalStorageUnit, filePath: 'test-dir/log.txt' },
   'My awesome log line\n',
);
```

## appendFile()

Append content into file directly:

- string support only
- append only

### Parameters

| Param | Type | Required | Description |
| ------ | -------- | :------: | ---- |
| `filePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |
| `content` | String | <div class="red">Yes</div> | Parsed content as string |


### Javascript Example

```javascript
await sos.fileSystem.appendFile(
   { storageUnit: internalStorageUnit, filePath: 'test-dir/log.txt' },
   'My awesome log line\n',
);
```

## readFile()
Read a text file.

::: alert alert--warning
**LG webOS limitation**:  
It's possible to handle only files up to **~10KB** on webOS devices. At the moment we are working on the solution regarding the file/s 'manipulation' on LG devices.
:::

### Parameters

| Param | Type | Required | Description |
| ----- | ---- | :-----: | ----- |
| `filePath` | FilePath | <div class="red">Yes</div> | Object contains storageUnit object returned by `listStorageUnits()` and `filePath` if the file or directory |

### Javascript Example
```javascript
// Written file can be read (coming soon)
const logFileContent = await sos.fileSystem.readFile({ storageUnit: internalStorageUnit, filePath: 'test-dir/log.txt' });
```

### Returns
```json
"content of your text file" // string
```

## Errors
Although we are doing our best, following errors may occur when working with the file system.

| Code | Type | Message |
| ---- | ---- | ------- |
| 50501 | InternalFileSystemError | Unexpected error occurred when listing storage units. |
| 50502 | InternalFileSystemError | Unexpected error occurred when listing files. |
| 50503 | InternalFileSystemError | Unexpected error occurred when checking for file existence. |
| 50504 | InternalFileSystemError | Unexpected error occurred when getting a file. |
| 50505 | InternalFileSystemError | Unexpected error occurred when writing a file. |
| 50506 | InternalFileSystemError | Unexpected error occurred when copying a file. |
| 50507 | InternalFileSystemError | Unexpected error occurred when moving a file. |
| 50508 | InternalFileSystemError | Unexpected error occurred when deleting a file. |
| 50509 | InternalFileSystemError | Unexpected error occurred when downloading a file. |
| 50510 | InternalFileSystemError | Unexpected error occurred when extracting a file. |
| 50511 | InternalFileSystemError | Unexpected error occurred when getting file checksum. |
| 50512 | InternalFileSystemError | Unexpected error occurred when creating directory. |
| 50513 | InternalFileSystemError | Unexpected error occurred when checking if a path is directory. |
| 50514 | InternalFileSystemError | Unexpected error occurred when reading a file. |
| 50515 | InternalFileSystemError | Unexpected error occurred when creating a link. |
| 50516 | InternalFileSystemError | Link is not supported on this platform. |
| 50518 | InternalFileSystemError | Unexpected error occurred when creating a archive file. |
