---
title:  Applet Resources
author: Miroslav Behina
date:   22.11.2018
type:   js-api
tags:
- applet
- applet_api
- api
- js_api
description: "[Content JS API] Often there are some JavaScripts and CSS styles necessary to be loaded for your Applet. It can be libraries like jQuery or even your whole application code. For these necessities offline API for saving, storing & loading files are ready."
---

# Applet Resources

Often there are some JavaScripts and CSS styles necessary to be loaded for your Applet. 
It can be libraries like jQuery or even your whole application code. For these necessities offline API for saving, storing & loading files are ready.


## All methods
::: table-responsive
| Methods | Description | Supported since |
| ------- | ----------- | ------ | :---------------: |
| `addFile()` | Add single file to Applet before starting it. | 1.0.0 |
| `addFiles()` | Add files to Applet before starting it. | 1.0.0 |
| `addFilesSync()` | Add files to Applet before starting it in the set order to prevent race condition. | 1.0.0 |
:::

### File object
Every method above, accept single object or array of objects depending by type. Every object **must have** these keys.

::: table-responsive
| Key | Type | Required | Description |
| :-----: | :-----: | :----: | ------- |
| `uri` | String | <div class="red">Yes</div> | URL of file |
| `uid` | String | <div class="red">Yes</div> | Unique ID in Applet for future manage |
| `type` | String | <div class="red">Yes</div> | Type of file. Example: `sos.offline.types.javascript`, `sos.offline.types.css`, `sos.offline.types.video` |
|||| ^^ |
| `flags` | Array | <div class="red">Yes</div> | Special flag for file |
| `headers` | Array | <div class="yellow">No</div> |  Authorization headers |
:::

::: alert alert--info
* The file URL must point to a file. If your URI leads to a 303 redirect (e.g. from http to http**s**), the API will not work.
* There are some limitations for [Emulator](https://docs.signageos.io/hc/en-us/articles/4405238997138). 
:::

## addFile()
Method `addFile()` will allow you to load single resource into applet. If you want to load more resource, use `addFiles()`.

### Javascript Example
```javascript
const file = { // File that will be loaded into an applet
    "uri": "https://ajax.googleapis.com/ajax/libs/jquery/2.2.4/jquery.min.js",
    "uid": "jquery-2.2.4.min.js",
    "type": sos.offline.types.javascript,
    "flags": [sos.offline.flags.append(document.body)]
}

await sos.offline.addFile(file); // And finally load file
```

## addFiles()
Method `addFiles()` will allow you to load array of your resources into Applet. Files are load in random order.

### Javascript Example
```javascript
const files = [ // Array of files
	{
		"uri": "https://ajax.googleapis.com/ajax/libs/jquery/2.2.4/jquery.min.js",
		"uid": "jquery-2.2.4.min.js",
		"type": sos.offline.types.javascript,
		"flags": [sos.offline.flags.append(document.body)]
	},
	{
		"uri": "https://signageostest.blob.core.windows.net/test/applet-examples/benchmark-styles.css?v=1.0.0",
		"uid": "benchmark-styles.css-v1.0.0",
		"type": sos.offline.types.css,
		"headers": { "Authorization": "Basic Zm9vOmJhcg==" },
		"flags": [sos.offline.flags.append(document.head)]
	}
];

await sos.offline.addFiles(files);
```

<div class="row d-flex align-content-stretch force-padding-20">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" target="_blank" href="https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/offline-resources">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fab fa-github svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Example of Applet that save files and load them in one file</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

## addFilesSync()
Method `addFilesSync()` works same as method above, but all files are loaded in correct order. Otherwise, files are loaded in random order by file size.

### Javascript example
```javascript
const files = [
	{
		"uri": "https://ajax.googleapis.com/ajax/libs/jquery/2.2.4/jquery.min.js",
		"uid": "jquery-2.2.4.min.js",
		"type": sos.offline.types.javascript,
		"flags": [sos.offline.flags.append(document.body)]
	},
	{
		"uri": "https://signageostest.blob.core.windows.net/test/applet-examples/benchmark-styles.css?v=1.0.0",
		"uid": "benchmark-styles.css-v1.0.0",
		"type": sos.offline.types.css,
		"headers": { "Authorization": "Basic Zm9vOmJhcg==" },
		"flags": [sos.offline.flags.append(document.head)]
	}
];

sos.offline.addFilesSync(files).then(() => {
	console.log('Now you can use jQuery and CSS file');
	jQuery('#do').something();
});
```

## Usage with Typescript
You can also use all these methods with [signageOS TypeScript](https://docs.signageos.io/hc/en-us/articles/4405069154962#signageos-typescript).

```typescript
interface ISaveFile {
	uid: string;
	uri: string;
	type: string;
	headers?: { [key: string]: string };
	flags?: IFlag[];
}

addFile(files: ISaveFile[]): Promise<void>;
addFiles(files: ISaveFile[]): Promise<void>;
addFilesSync(files: ISaveFile[]): Promise<void>;
```

Flags that can be optionally passed to the file object are available in flags property.

```typescript
interface IFlag {
	type: 'append';
	element: IElement;
}
```

## Errors
Although we are doing our best, following errors may occur when working with the Applet Resources.

::: table-responsive
| Code | Type | Message |
| ---- | ---- | ------- |
| 40101 | AppletResourcesError | File needs to be object |
| 40102 | AppletResourcesError | File UID needs to be defined |
| 40103 | AppletResourcesError | File flags needs to be Array |
| 40104 | AppletResourcesError | File flag needs to be object |
| 40105 | AppletResourcesError | Already loading file: `FILE_NAME` |
|^^|^^| Your code has probably multiple occurrences of addFile/s methods. |
| 40106 | AppletResourcesError | Already existing file: `FILE_NAME` |
|^^|^^| File is already loaded or exists in Applet |
| 49901 | FileNotFoundError    | File `FILE_ID` was not found |
|^^|^^| Applet can not get file. URL is invalid. |
:::
