"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.capitalize = capitalize;
exports.uncapitalize = uncapitalize;
exports.getFirstCamelCaseWord = getFirstCamelCaseWord;
exports.getFirstPascalCaseWord = getFirstPascalCaseWord;
exports.getFirstWord = getFirstWord;
exports.toPascalCase = toPascalCase;
exports.toCamelCase = toCamelCase;
function capitalize(word) {
    return (word.charAt(0).toUpperCase() + word.slice(1));
}
function uncapitalize(word) {
    return (word.charAt(0).toLowerCase() + word.slice(1));
}
/**
 * Get first word in camelCase string.
 * If it is not camelCase, it returns null.
 */
function getFirstCamelCaseWord(varName) {
    // when starts with lowercase (is camelCase)
    const matches = varName.match(/^([a-z][a-z0-9]*)([A-Z]|$)/);
    if (matches) {
        return matches[1];
    }
    return null;
}
/**
 * Get first word in PascalCase string.
 * If it is not PascalCase, it returns null.
 */
function getFirstPascalCaseWord(varName) {
    // when starts with uppercase (is PascalCase)
    const wordMatches = varName.match(/^([A-Z][a-z0-9]+)([A-Z]|$)/);
    if (wordMatches) {
        return wordMatches[1];
    }
    const shortcutMatches = varName.match(/^([A-Z]+)(([A-Z][a-z0-9])|$)/);
    if (shortcutMatches) {
        return shortcutMatches[1];
    }
    return null;
}
/**
 * Combines functions `getFirstCamelCaseWord` and `getFirstPascalCaseWord`.
 * It detects input variable name format and returns first word.
 * If it is not camelCase or PascalCase, it returns null.
 */
function getFirstWord(varName) {
    if (varName.charAt(0).toUpperCase() === varName.charAt(0)) {
        return getFirstPascalCaseWord(varName);
    }
    if (varName.charAt(0).toLowerCase() === varName.charAt(0)) {
        return getFirstCamelCaseWord(varName);
    }
    return null;
}
/**
 * Declares how long words are stated as shortcuts.
 * This is used to determine if a word is a shortcut or not.
 * Incorrect camelCase/PascalCase words use capital or uncapital shortcuts.
 * E.g.:
 * "UI" in camelCase is "ui", in PascalCase is "UI".
 * However, correctly, it should be "ui" and "Ui" instead.
 */
const DEFAULT_MAX_SHORTCUT_LENGTH = 3;
/**
 * Converts camelCase/PascalCase to PascalCase
 * The argument has to be one of the following:
 * PascalCase
 * camelCase
 */
function toPascalCase(camelVarName, maxShortcutLength = DEFAULT_MAX_SHORTCUT_LENGTH) {
    const firstWord = getFirstWord(camelVarName);
    if (!firstWord) {
        return null;
    }
    const pascalFirstWord = firstWord.length > maxShortcutLength ? capitalize(firstWord) : firstWord.toUpperCase();
    return pascalFirstWord + camelVarName.slice(pascalFirstWord.length);
}
/**
 * Converts PascalCase/camelCase to camelCase
 * The argument has to be one of the following:
 * PascalCase
 * camelCase
 */
function toCamelCase(pascalVarName, maxShortcutLength = DEFAULT_MAX_SHORTCUT_LENGTH) {
    const firstWord = getFirstWord(pascalVarName);
    if (!firstWord) {
        return null;
    }
    const camelFirstWord = firstWord.length > maxShortcutLength ? uncapitalize(firstWord) : firstWord.toLowerCase();
    return camelFirstWord + pascalVarName.slice(camelFirstWord.length);
}
//# sourceMappingURL=word.js.map