import { Server as HttpServer } from 'http';
import { Application } from 'express';
import { IMetricsClient } from '../Metrics/MetricsClient';
export interface IHttpServer {
    /** Main HTTP server */
    server: HttpServer;
    /** Use this to extend application HTTP routes */
    express: Application;
    /** Monitoring runs on a separate port for security reasons */
    monitoringServer: HttpServer;
    /** Use this to extend monitoring HTTP routes */
    monitoringExpress: Application;
    listen(): Promise<void>;
    /**
     * Waits for the server to finish serving all requests, then closes it
     */
    close(): Promise<void>;
}
export type OnServiceStopCallback = () => Promise<void>;
export interface IService {
    httpServer: IHttpServer;
    metricsClient: IMetricsClient;
    /**
     * Returns true if the service has been started
     */
    isStarted(): boolean;
    /**
     * Starts the service and all its sub-services, i.e. the HTTP server
     */
    start(): Promise<void>;
    /**
     * Stops the service and all the sub-services managed by it, i.e. the HTTP server
     *
     * It will block until HTTP server finishes serving all requests
     */
    stop(): Promise<void>;
    /**
     * Add a custom listener for when the service is stopped
     *
     * This callback is called when the service receives a stop signal or when the service method stop() is called.
     *
     * This callback should perform any necessary cleanup before the service and the entire process is stopped.
     * Typically that would include:
     *   - draining connections
     *   - waiting for all pending requests to finish
     *   - calling external services to notify them about the shutdown
     *
     * @note database connections are automatically closed when the service is stopped.
     */
    on(event: 'stop', callback: OnServiceStopCallback): this;
    /**
     * Remove a custom listener previously added with on()
     */
    off(event: 'stop', callback: OnServiceStopCallback): this;
}
