"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compose = compose;
/**
 * Composes multiple functions into a single function.
 *
 * The functions are composed from right to left, meaning that the first function in the list will be the last to be called.
 *
 * For example:
 *   compose(f, g, h)(x) is equivalent to f(g(h(x)))
 *
 * @param functions The functions to compose
 * @returns A single function that is the composition of all the provided functions
 *
 * @example
 *   ```typescript
 *   const addOne = (x: number) => x + 1;
 *   const multiplyByTwo = (x: number) => x * 2;
 *   const addOneAndMultiplyByTwo = compose(multiplyByTwo, addOne);
 *   const result = addOneAndMultiplyByTwo(3); // 8
 *   ```
 */
function compose() {
    var functions = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        functions[_i] = arguments[_i];
    }
    if (functions.length === 0) {
        throw new Error('compose requires at least one argument');
    }
    if (functions.length === 1) {
        return functions[0];
    }
    return functions.reduceRight(function (acc, f) {
        return function () {
            var args = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                args[_i] = arguments[_i];
            }
            return f(acc.apply(void 0, args));
        };
    });
}
//# sourceMappingURL=compose.js.map