"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createEventProcessingMetrics = createEventProcessingMetrics;
const defaultOptions = {
    /**
     * Buckets for the event_duration_seconds_bucket metric.
     * Default buckets are [0.01, 0.05, 0.1, 0.5, 1, 2, 5, 10, 30, 60]
     */
    durationInSecondsHistogramBuckets: [0.01, 0.05, 0.1, 0.5, 1, 2, 5, 10, 30, 60],
};
function createEventProcessingMetrics(metricsClient, options) {
    const client = metricsClient.getClient();
    options = Object.assign(Object.assign({}, defaultOptions), options);
    const eventCount = new client.Counter({
        name: 'event_consumer_events_total',
        help: 'Number of all events',
        labelNames: ['eventType'],
    });
    const eventDurationSum = new client.Counter({
        name: 'event_consumer_event_duration_seconds_total',
        help: 'Total duration of all events in seconds',
        labelNames: ['eventType', 'phase'],
    });
    const eventDurationHistogram = new client.Histogram({
        name: 'event_consumer_event_duration_seconds',
        help: 'Duration of all events in seconds',
        buckets: options.durationInSecondsHistogramBuckets,
    });
    function recordMetrics({ event, stats }) {
        eventCount.labels({ eventType: event.type }).inc();
        eventDurationHistogram.observe(stats.duration / 1e3);
        eventDurationSum.labels({ eventType: event.type, phase: 'processing' }).inc(stats.duration / 1e3);
        const eventStartDuration = stats.start - event.receivedAt.valueOf();
        eventDurationSum.labels({ eventType: event.type, phase: 'eventStart' }).inc(Math.max(eventStartDuration, 0) / 1e3);
        const commandStartDuration = event.receivedAt.valueOf() - event.dispatchedAt.valueOf();
        if (commandStartDuration !== 0) {
            eventDurationSum.labels({ eventType: event.type, phase: 'commandStart' }).inc(Math.max(commandStartDuration, 0) / 1e3);
        }
    }
    return {
        recordMetrics,
        eventCount,
        eventDurationSum,
        eventDurationHistogram,
    };
}
//# sourceMappingURL=amqpEventConsumerExporter.js.map