"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmqpEventConsumer = void 0;
const lockedDecorator_1 = require("../../../Lock/lockedDecorator");
const logging_1 = require("../../../Logging/logging");
const eventQueue_1 = require("../eventQueue");
const amqpEventConsumerExporter_1 = require("../../../Metrics/amqpEventConsumerExporter");
const consumerLogger = logging_1.logger.createNamespace('EventConsumer');
const DEFAULT_DOMAIN = 'default';
const LOCK_NAMESPACE = 'AmqpEventConsumer';
/**
 * High level abstraction of event consumption
 *
 * This implementation uses AMQP queues to consume events.
 *
 * It abstracts all the implementation details of setting up consumption of events
 * and provides a simple interface so that the consumer can focus on just handling the events.
 *
 * All events are consumed from a single queue in a sequential order.
 * The name of the queue will be constructed from the consumer name and the domain as "<consumerName>_<domain>".
 */
class AmqpEventConsumer {
    /**
     * @param amqpConnection The AMQP connection to use
     * @param consumerName The name of the consumer. This is used to identify the consumer in the AMQP queue
     * @param domain The domain to consume events from. Defaults to 'default'
     * @param options Additional options for the consumer
     */
    constructor({ amqpConnection, consumerName, domain, options, metricsClient, }) {
        this.consumerCallbacks = {};
        this.stopCallback = null;
        this.started = false;
        this.amqpConnection = amqpConnection;
        this.consumerName = consumerName;
        this.domain = domain || DEFAULT_DOMAIN;
        this.options = options;
        this.metrics = metricsClient ? (0, amqpEventConsumerExporter_1.createEventProcessingMetrics)(metricsClient) : undefined;
    }
    /**
     * Add a consumer for a specific event type
     *
     * Has to be called before calling listen(), otherwise it will throw an error.
     */
    addEventConsumer(eventType, callback, option) {
        if (this.consumerCallbacks[eventType]) {
            throw new Error(`Consumer for event type ${eventType} already registered`);
        }
        if (this.started) {
            throw new Error('Cannot add event consumer after consumer has started');
        }
        this.consumerCallbacks[eventType] = {
            callback,
            schema: option === null || option === void 0 ? void 0 : option.schema,
        };
    }
    listen() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.started) {
                throw new Error('Event consumer already started');
            }
            yield this.bindEvents();
            this.started = true;
        });
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.stopCallback) {
                const closeChannel = yield this.stopCallback();
                yield closeChannel();
                this.stopCallback = null;
            }
            this.started = false;
        });
    }
    bindEvents() {
        return __awaiter(this, void 0, void 0, function* () {
            const eventTypes = Object.keys(this.consumerCallbacks);
            this.stopCallback = yield (0, eventQueue_1.bindMany)(this.amqpConnection, eventTypes, this.domain, this.consumerName, this.consumeEvent.bind(this), this.options);
        });
    }
    shouldConsumeEvent(options) {
        if (options.schema) {
            return options.schema.safeParse(options.event).success;
        }
        return true;
    }
    consumeEvent(event) {
        return __awaiter(this, void 0, void 0, function* () {
            const { callback, schema } = this.consumerCallbacks[event.type] || {};
            if (!callback) {
                throw new Error(`Event ${event.type} received but no consumer registered`);
            }
            if (this.shouldConsumeEvent({ schema, event })) {
                const start = new Date().getTime();
                yield callback(event);
                if (this.metrics) {
                    this.metrics.recordMetrics({
                        event,
                        stats: { start, duration: new Date().getTime() - start },
                    });
                }
                consumerLogger.info(`Event ${event.type} consumed`, { event });
            }
        });
    }
}
exports.AmqpEventConsumer = AmqpEventConsumer;
__decorate([
    (0, lockedDecorator_1.locked)(LOCK_NAMESPACE, { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], AmqpEventConsumer.prototype, "listen", null);
__decorate([
    (0, lockedDecorator_1.locked)(LOCK_NAMESPACE, { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], AmqpEventConsumer.prototype, "close", null);
//# sourceMappingURL=AmqpEventConsumer.js.map