import { S3ClientConfig } from '@aws-sdk/client-s3';
import { BucketExistsArgs, CreateBucketArgs, CreateBucketResponse, DeleteBucketArgs, DeleteFileArgs, DeleteObjectArgs, DownloadFileArgs, GetMetadataArgs, GetObjectArgs, GetObjectResponse, GetPresignedUrlArgs, HeadObject, ListFilesArgs, PutBucketPolicyArgs, UploadContentArgs, UploadFileArgs, UploadStreamArgs } from './S3Types';
/**
 * Wrapper for AWS S3 client
 *
 * All commonly used operations with S3 will be implemented here to provide a consistent interface across the codebase.
 */
export interface IAwsS3 {
    /**
     * Upload a local file to S3
     */
    uploadFile(args: UploadFileArgs): Promise<void>;
    /**
     * Upload contents of a file stored in memory directly to S3
     */
    uploadContent(args: UploadContentArgs): Promise<void>;
    /**
     * Upload stream to S3
     */
    uploadStream(args: UploadStreamArgs): Promise<void>;
    /**
     * Download a file from S3 to local file system
     */
    downloadFile({ bucket, remotePath, filePath }: DownloadFileArgs): Promise<void>;
    /**
     * List files in a particular path in a bucket
     */
    listFiles({ bucket, remotePath }: ListFilesArgs): Promise<string[]>;
    /**
     * Delete a file from S3
     */
    deleteFile(args: DeleteFileArgs): Promise<void>;
    /**
     * Get object head
     */
    getObjectHead({ bucket, remotePath }: GetMetadataArgs): Promise<HeadObject>;
    /**
     * Gets an object from a bucket
     */
    getObject({ bucket, key }: GetObjectArgs): Promise<GetObjectResponse>;
    /**
     * Removes an object from a bucket
     */
    deleteObject({ bucket, key }: DeleteObjectArgs): Promise<void>;
    /**
     * Creates a S3 bucket
     */
    createBucket({ bucket }: CreateBucketArgs): Promise<CreateBucketResponse>;
    /**
     * Removes a S3 bucket
     */
    deleteBucket({ bucket }: DeleteBucketArgs): Promise<void>;
    /**
     * Checks if S3 bucket exists
     */
    bucketExists({ bucket }: BucketExistsArgs): Promise<boolean>;
    /**
     * Sets a permission policy on a S3 bucket
     */
    putBucketPolicy({ bucket, policy }: PutBucketPolicyArgs): Promise<void>;
    /**
     * Get a pre-signed URL for an object in a bucket
     * @param bucket - bucket name
     * @param remotePath - path to the object in the bucket
     * @param expiration - URL expiration time in seconds
     */
    getPresignedUrl({ bucket, remotePath, expirationInSec }: GetPresignedUrlArgs): Promise<string>;
}
/**
 * Wrapper for AWS S3 client
 *
 * All commonly used operations with S3 will be implemented here to provide a consistent interface across the codebase.
 */
export declare class AwsS3 implements IAwsS3 {
    private client;
    /**
     * @param config.forcePathStyle - Whether to use path style URLs for S3 requests.
     * Defaults to the value of the `AWS_S3_FORCE_PATH_STYLE_URLS` environment variable or false if not set.
     */
    constructor(config?: S3ClientConfig);
    uploadFile({ filePath, bucket, remotePath, contentType, metadata }: UploadFileArgs): Promise<void>;
    uploadContent({ content, bucket, remotePath, contentType, metadata }: UploadContentArgs): Promise<void>;
    uploadStream({ bucket, remotePath, stream }: UploadStreamArgs): Promise<void>;
    downloadFile({ bucket, remotePath, filePath }: DownloadFileArgs): Promise<void>;
    listFiles({ bucket, remotePath, maxKeys, startAfterKey }: ListFilesArgs): Promise<string[]>;
    deleteFile({ bucket, remotePath }: DeleteFileArgs): Promise<void>;
    getObjectHead({ bucket, remotePath }: GetMetadataArgs): Promise<HeadObject>;
    getObject({ bucket, key }: GetObjectArgs): Promise<GetObjectResponse>;
    deleteObject({ bucket, key }: DeleteObjectArgs): Promise<void>;
    createBucket({ bucket }: CreateBucketArgs): Promise<CreateBucketResponse>;
    deleteBucket({ bucket }: DeleteBucketArgs): Promise<void>;
    bucketExists({ bucket }: BucketExistsArgs): Promise<boolean>;
    putBucketPolicy({ bucket, policy }: PutBucketPolicyArgs): Promise<void>;
    getPresignedUrl({ bucket, remotePath, expirationInSec }: GetPresignedUrlArgs): Promise<string>;
}
