import { Originator } from '../Originator/Originator';
import { DateTimeSnoozeRule, SnoozeRule } from './AlertSnooze';
export interface AlertIdentifiable {
    /** Unique identifier for each alert across system */
    alertUid: string;
}
export interface AlertCreatable extends AlertIdentifiable {
    /**
     * Alert is always related to single organization in time of creation.
     * There should not exists more than one alert for single organization with same alertRuleUid.
     */
    organizationUid: string;
    /** Detailed description of alert by default generated by system. It can be changed by user later on. */
    description: string;
    /** Relation to the AlertRule which is originator of alert creation. */
    alertRuleUid: string;
}
export interface AlertUpdatable extends AlertIdentifiable {
    /** User adjusted description of alert */
    description: string;
}
export interface AlertDeviceAssignable extends AlertIdentifiable {
    /** device identity assigned to single alert */
    deviceIdentityHash: string;
}
export interface AlertSnoozable extends AlertIdentifiable {
    snoozeRule: SnoozeRule;
}
type Nullable<T> = {
    [P in keyof T]: T[P] | null;
};
export interface IAlertHistoryItem {
    /**
     * Newly set values. Empty values were not changed.
     * Some values are always omitted to prevent data usage.
     */
    values: Partial<Omit<Alert, 'history' | 'deviceIdentityHashes'> & {
        snoozedDevice?: {
            deviceIdentityHash: string;
            snoozeRule: unknown;
        };
        unsnoozedDevice?: {
            deviceIdentityHash: string;
            unsnoozedAt: Date;
        };
        unassignedDevice?: {
            deviceIdentityHash: string;
            unassignedAt: Date;
        };
    }>;
    /** Date of update in history */
    updatedAt: Date;
    originator: Originator;
}
export interface AlertEmailNotifiable {
    alertUid: string;
    emailAddresses: string[];
    description: string;
    affectedDevices: number;
    totalDevices: number;
    organization: string;
    alertRuleName: string;
    organizationUid: string;
    alertRuleUid: string;
    whiteLabelSettingsUid?: string;
}
type SnoozedDevice = {
    deviceIdentityHash: string;
    snoozeRule: DateTimeSnoozeRule;
};
/**
 * The Alert rule entity contains all properties.
 * All properties are required except the null ones.
 * Undefined value is not used because it's reserved for no changes in update operation.
 * Single rule is a definition set which will be used for creation of specific alerts in time.
 */
export type Alert = AlertCreatable & Required<AlertUpdatable> & Nullable<AlertSnoozable> & {
    /**
     * All assigned devices into the alert.
     * It's aggregation on all assign & unassign operations in time.
     */
    deviceIdentityHashes: string[];
    /** When the entity was created */
    createdAt: Date;
    /** The last change date of updates assigned/unassigned devices. Not user interactions. */
    latelyChangedAt: Date;
    /** The last change date of alert manager check. Not user interactions. */
    latelyProcessedAt?: Date;
    /** All history of changes of alert by user. */
    history: IAlertHistoryItem[];
    /** Indicates whether alert is currently archived or not. */
    archivedAt: Date | null;
    snoozedDevices?: SnoozedDevice[];
};
export {};
