"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createCommandFactory = createCommandFactory;
const commandSchemas_1 = require("./commandSchemas");
const zodError_1 = require("@signageos/lib/dist/Zod/zodError");
/**
 * This is abstraction of the command factory. It should be used to create command factory for each command type.
 *
 * All command types should create it's own command factory using this function.
 * It has to be created from existing zod schema. And optionally from property generators.
 * Optional property generators are used to generate some specific properties automatically.
 * For example it can generate random UIDs for the command (it can be nested recursively in the payload).
 *
 * E.g.:
 * ```ts
 * export const CreateOrganization = "Organization.CreateOrganization";
 * export const CreateOrganizationSchema = z.strictObject({
 * 	type: z.literal(CreateOrganization),
 * 	uid: UIDSchema,
 * 	// ...
 * });
 * export type CreateOrganization = z.infer<typeof CreateOrganizationSchema>;
 * export const CreateOrganizationFactory = createCommandFactory(CreateOrganizationSchema, { uid });
 * ```
 *
 * @param schema Zod schema of the command.
 * @param generators See the PropGenerators type for more info.
 * @returns Factory function for generating commands based on given schema and generators.
 */
function createCommandFactory(schema, generators = {}, options) {
    if (!(options === null || options === void 0 ? void 0 : options.noRegisterCommandSchema)) {
        (0, commandSchemas_1.registerCommandSchema)(schema);
    }
    return (data) => {
        try {
            const type = (0, commandSchemas_1.getCommandTypeOfSchema)(schema);
            const command = Object.assign({ type }, data);
            generateRecursivelyMutable(command, generators);
            return schema.parse(command);
        }
        catch (error) {
            throw (0, zodError_1.formatIfZodError)(error);
        }
    };
}
/**
 * It will go deeply through the object and generate all values from given generators.
 * It uses the original object and modifies it in place (mutable).
 * It will not generate values for the fields that are already defined.
 */
function generateRecursivelyMutable(thisPayload, thisGenerators) {
    for (let field in thisGenerators) {
        const generator = thisGenerators[field];
        if (typeof generator === 'function') {
            if (thisPayload[field] === undefined) {
                thisPayload[field] = generator(thisPayload);
            }
        }
        else if (generator !== undefined) {
            generateRecursivelyMutable(thisPayload[field], generator);
        }
    }
}
//# sourceMappingURL=commandFactory.js.map