import { z, ZodTypeAny } from 'zod';
import { OriginatorAwareCommandSchema } from '../ICommand';
/**
 * Basic entity schema that all commands that work with an entity should extend
 */
export declare const EntitySchema: z.ZodObject<{
    uid: z.ZodString;
}, "strict", z.ZodTypeAny, {
    uid: string;
}, {
    uid: string;
}>;
type EntitySchemaShape = typeof EntitySchema['shape'];
type OriginatorAwareCommandSchemaShape = typeof OriginatorAwareCommandSchema['shape'];
type BaseEntityCommandShape<Type extends string> = {
    type: z.ZodLiteral<Type>;
} & EntitySchemaShape & OriginatorAwareCommandSchemaShape;
/**
 * Basic command to create an entity that commands for specific entities should extend
 */
export type CreateEntityCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape> = z.ZodObject<BaseEntityCommandShape<Type> & TShape, 'strict'>;
export type CreateEntitiesCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape> = z.ZodObject<BaseEntityCommandShape<Type> & {
    items: z.ZodArray<z.ZodObject<TShape>>;
}, 'strict'>;
/**
 * Basic command to update an entity that commands for specific entities should extend
 */
export type UpdateEntityCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape> = z.ZodObject<BaseEntityCommandShape<Type> & {
    [K in keyof TShape]: TShape[K] extends z.ZodOptional<ZodTypeAny> ? z.ZodNullable<TShape[K]> : z.ZodOptional<TShape[K]>;
}, 'strict'>;
/**
 * Basic command to delete an entity that commands for specific entities should extend
 */
export type DeleteEntityCommandSchema<Type extends string> = z.ZodObject<BaseEntityCommandShape<Type>, 'strict'>;
/**
 * Takes entity schema and generates a schema of a create command for that entity
 *
 * Here's how the create command schema is generated from the original entity schema:
 *   * the entire entity schema is kept as is
 *   * type literal equal to "type" arg is added
 *   * OriginatorAwareCommandSchema is added
 */
export declare function createCreateEntityCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape>(type: Type, schema: z.ZodObject<TShape, 'strict'>): CreateEntityCommandSchema<Type, TShape>;
/**
 *
 * @param type - type literal of a command
 * @param schema - schema to validate against
 * @returns - new schema with an array of entities
 */
export declare function createCreateEntitiesCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape>(type: Type, schema: z.ZodObject<TShape, 'strict'>): CreateEntitiesCommandSchema<Type, TShape>;
/**
 * Takes entity schema and generates a schema of an update command for that entity
 *
 * Here's how the update command schema is generated from the original entity schema:
 *   * uid stays the same
 *   * type literal equal to "type" arg is added
 *   * OriginatorAwareCommandSchema is added
 *   * all other fields are changed to optional so that any subset of the entity's fields can be updated
 *   * all fields that are optional in the original entity schema are changed to optional or null
 *     where null means that the fields would be unset (it's optional so it should be possible to unset it)
 */
export declare function createUpdateEntityCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape>(type: Type, schema: z.ZodObject<TShape, 'strict'>): UpdateEntityCommandSchema<Type, TShape>;
/**
 * Takes entity schema and generates a schema of a delete command for that entity
 *
 * Here's how the create command schema is generated from the original entity schema:
 *   * only uid is taken from the schema since that's all that's necessary to reference the entity
 *   * type literal equal to "type" arg is added
 *   * OriginatorAwareCommandSchema is added
 */
export declare function createDeleteEntityCommandSchema<Type extends string, TShape extends EntitySchemaShape & z.ZodRawShape>(type: Type, schema: z.ZodObject<TShape, 'strict'>): DeleteEntityCommandSchema<Type>;
export {};
