"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockedEventConsumer = void 0;
const eventQueue_1 = require("../eventQueue");
const debug_1 = __importDefault(require("debug"));
const deferredState_1 = require("./deferredState");
const withLock_1 = require("./withLock");
const debug = (0, debug_1.default)('@signageos/lib:AMQP:EventSourcing:Locked:LockedEventConsumer');
/**
 * The default values for the bind options doesn't allow concurrent processing of events.
 */
const DEFAULT_BIND_OPTIONS = {
    mainPrefetchCount: 1,
    distributedPrefetchCount: 0,
    redeliverDelayMs: 30e3,
    legacyNotification: false,
};
const mainEventQueueBindOptions = {
    queueType: 'quorum',
    singleActiveConsumer: true,
    suppressFirstError: false,
    redeliverDelayMs: 0,
    deadLetterIfErred: true,
};
const distributedEventQueueBindOptions = {
    queueType: 'classic',
    singleActiveConsumer: false,
    suppressFirstError: false,
    redeliverDelayMs: 0,
};
const DEFAULT_LOCK_KEY = '$default';
/**
 * Event consumer that consumes events in a locked manner.
 *
 * It ensures that only one event with the same lockKey is processed at a time.
 * Practically, it means that events with the same lockKey are processed sequentially.
 * However, events with different lockKeys can be processed in parallel.
 *
 * Internal implementation uses 2 separate queues (if distributedPrefetchCount > 0):
 * - Main queue for distributing events to distributed queue
 * - Distributed queue for processing events
 *
 * The main queue is "Single Active Consumer" and is locking the event processing based on lockKeys field of the event.
 * So the order of events is preserved based on the lockKeys.
 * However, the distributed queue is not locking the event processing and is processing across multiple consumers at the same time.
 * Messages in the distributed queue are transient. In case of failure, they are not lost because the main queue is persistent
 * and is not acked until the distributed queue acks the message.
 */
class LockedEventConsumer {
    constructor(amqpConnection) {
        this.amqpConnection = amqpConnection;
        /*
        TODO this lock is not rejecting newly coming events, but only those that are currently waiting in the DeferredState.
        The right implementation should reject all events that are coming even after the first event failed.
        Otherwise, the dead-lettered events will be consumed later than the new events.
        So the order of events will be broken.
        Short-term solution would be disabling `deadLetterIfErred` option for the main queue.
        However, it would produce infinite redeliveries of the same events and rejecting them again and again.
        Other solution would be to reject all events that are coming after the first failed event
        and on top of that persist the DeferredState somewhere to be able to reject them even after the instance is restarted.
        */
        this.withLock = (0, withLock_1.createWithLock)(new deferredState_1.DeferredState());
    }
    bind(eventTypes_1, domainName_1, consumerType_1, onEvent_1) {
        return __awaiter(this, arguments, void 0, function* (eventTypes, domainName, consumerType, onEvent, bindOptions = DEFAULT_BIND_OPTIONS) {
            var _a;
            bindOptions = Object.assign(Object.assign({}, DEFAULT_BIND_OPTIONS), bindOptions);
            yield (0, eventQueue_1.prepareRejected)(this.amqpConnection, domainName, consumerType, {
                redeliverDelayMs: (_a = bindOptions.redeliverDelayMs) !== null && _a !== void 0 ? _a : DEFAULT_BIND_OPTIONS.redeliverDelayMs,
            });
            const mainBindOptions = this.createMainBindOptions(bindOptions);
            const distributedBindOptions = this.createDistributedBindOptions(bindOptions);
            const consumeEvent = this.createConsumeEvent(domainName, consumerType, mainBindOptions, onEvent);
            const cancelConsumptions = [];
            if (bindOptions.distributedPrefetchCount === 0) {
                // If distributed consumption is disabled, consume events directly
                const cancelMain = yield (0, eventQueue_1.bindMany)(this.amqpConnection, eventTypes, domainName, consumerType, consumeEvent, mainBindOptions);
                cancelConsumptions.push(cancelMain);
            }
            else {
                // If distributed consumption is enabled, consume events using main queue and distribute them to distributed queue
                const distributeEvent = this.createDistributeEvent(domainName, consumerType, distributedBindOptions);
                const cancelMain = yield (0, eventQueue_1.bindMany)(this.amqpConnection, eventTypes, domainName, consumerType, distributeEvent, mainBindOptions);
                const cancelDistributed = yield (0, eventQueue_1.bindRPC)(this.amqpConnection, domainName, consumerType, consumeEvent, distributedBindOptions);
                cancelConsumptions.push(cancelMain);
                cancelConsumptions.push(cancelDistributed);
            }
            return () => __awaiter(this, void 0, void 0, function* () {
                const closeChannels = yield Promise.all(cancelConsumptions.map((cancel) => cancel()));
                return () => __awaiter(this, void 0, void 0, function* () {
                    yield Promise.all(closeChannels.map((close) => close()));
                });
            });
        });
    }
    createMainBindOptions(bindOptions) {
        var _a;
        return Object.assign(Object.assign({}, mainEventQueueBindOptions), { prefetchCount: bindOptions.mainPrefetchCount, notification: (_a = bindOptions.legacyNotification) !== null && _a !== void 0 ? _a : false });
    }
    createConsumeEvent(domainName, consumerType, bindOptions, onEvent) {
        return (event) => __awaiter(this, void 0, void 0, function* () {
            const logMetadata = {
                domainName,
                consumerType,
                bindOptions,
                event,
            };
            debug(`Consuming event ${event.type} from ${domainName}.${consumerType}`, logMetadata);
            yield onEvent(event);
            debug(`Consumed event ${event.type} from ${domainName}.${consumerType}`, logMetadata);
        });
    }
    createDistributedBindOptions(bindOptions) {
        return Object.assign(Object.assign({}, distributedEventQueueBindOptions), { prefetchCount: bindOptions.distributedPrefetchCount });
    }
    createDistributeEvent(domainName, consumerType, bindOptions) {
        return (event) => __awaiter(this, void 0, void 0, function* () {
            var _a;
            const logMetadata = {
                domainName,
                consumerType,
                bindOptions,
                event,
            };
            debug(`Received event ${event.type} to ${domainName}.${consumerType}`, logMetadata);
            const lockKeys = (_a = event.lockKeys) !== null && _a !== void 0 ? _a : [DEFAULT_LOCK_KEY]; // TODO default lock could be removed one day in future when all events are migrated
            yield this.withLock(lockKeys, () => __awaiter(this, void 0, void 0, function* () {
                debug(`Distributing event ${event.type} to ${domainName}.${consumerType}`, logMetadata);
                // TODO Add timeout for processing event, because it can be stucked until the instance is restarted
                yield (0, eventQueue_1.processRPC)(this.amqpConnection, domainName, consumerType, event, { persistent: false });
            }));
            debug(`Distributed event consumed ${event.type} to ${domainName}.${consumerType}`, logMetadata);
        });
    }
}
exports.LockedEventConsumer = LockedEventConsumer;
//# sourceMappingURL=lockedEventConsumer.js.map