import { ZodError, ZodIssue } from 'zod';
/**
 * This is the same shape as ZodError but came from external source (e.g. command-handler)
 * So it can be even plain object.
 */
export type ZodLikeError = {
    name: string;
    issues: ZodIssue[];
};
/**
 * This kind of error (or plain object) can come from external service (e.g. command-handler)
 * The originalError can be anything.
 */
export type ExternalLikeError<T> = Error & {
    originalError: T;
};
/**
 * Detect whether the object is a ZodError (instance of ZodError could not work)
 */
export declare function isZodError(error: unknown | Partial<ZodLikeError>): error is ZodError;
/**
 * Detect whether the object is an external error with Zod-like originalError inside
 */
export declare function isExternalZodError(error: unknown | Partial<ExternalLikeError<Partial<ZodLikeError>>>): error is ExternalLikeError<ZodLikeError>;
export declare const formatZodErrorMessage: (error: ZodError) => string;
export declare class FormattedZodError extends Error {
    readonly zodError: ZodError;
    constructor(message: string, zodError: ZodError);
}
/**
 * Returns the alternative instance of FormettedZodError
 * that contains the original zodError but the message
 * is formatted as a human readable single string.
 */
export declare const formatZodError: (error: ZodError) => FormattedZodError;
/**
 * Format only if the error is ZodError
 * @see formatZodError
 */
export declare const formatIfZodError: (error: unknown) => unknown;
