Param(
  [Parameter(Mandatory = $True)]
  [string]$upgradeArchiveUrl
)

$isDebug = $DebugPreference -ne "SilentlyContinue"

function Invoke-RestartComputer() {
  Restart-Computer -Force -WhatIf:$isDebug
  if ($isDebug) {
    Exit
  }
}

. "$PSScriptRoot\Upgrade-ServerHelperFunctions.ps1"

$kioskUsername = 'Kiosk01'
$baseDirPath = Resolve-Path -Path "\signageos"

$serverDirName = "server"
$serverDirPath = (Join-Path $baseDirPath $serverDirName)

$serverBackupDirName = "server_backup"
$serverBackupDirPath = (Join-Path $baseDirPath $serverBackupDirName)

$upgradeDirName = "upgrade"
$upgradeDirPath = (Join-Path $baseDirPath $upgradeDirName)
$upgradeArchiveName = "sos_upgrade.zip"
$upgradeArchivePath = (Join-Path $upgradeDirPath $upgradeArchiveName)
$upgradeServerArchiveName = "2-SignageOS_Server\SignageOS-Server.zip"
$upgradeServerArchivePath = (Join-Path $upgradeDirPath $upgradeServerArchiveName)
$upgradeServerDirPath = (Join-Path $upgradeDirPath "2-SignageOS_Server\SignageOS-Server")
$upgradeFailedDirName = "server_failed_upgrade"
$upgradeFailedDirPath = (Join-Path $baseDirPath $upgradeFailedDirName)

$daemonServiceId = "signageosserver.exe"
$daemonScreenshotServiceId = "signageosscreenshotserver.exe"
$daemonService = (Join-Path $serverDirPath "daemon\$daemonServiceId")
$daemonScreenshotService = (Join-Path $serverDirPath "dist\screenshotServer\daemon\$daemonScreenshotServiceId")

Try {
  Start-Transcript -Path (Join-Path $baseDirPath "\logs\upgrade_log.txt") -Force
} Catch {
  Start-Transcript -Path (Join-Path $baseDirPath "\logs\upgrade_log.txt") -Force
}

Try {
  .$daemonService stopwait
  .$daemonScreenshotService stopwait
} Catch {
  Write-ErrorToHost "Upgrade failed on stopping signageos services. Server files are intact."
  Write-ErrorToHost ($ErrorFormatString -F $_.Exception.Message, $_.InvocationInfo.PositionMessage, $_.CategoryInfo.ToString(), $_.FullyQualifiedErrorId)

  Invoke-RestartComputer
}
Write-Debug "Services stopped."

Try {
  Remove-DirItems -Path $serverBackupDirPath
  Remove-DirItems -Path $upgradeDirPath

  New-Item -Path $baseDirPath -Name $serverBackupDirName -ItemType Directory -Force | Out-Null
  Copy-Item -Path "$serverDirPath\*" -Destination $serverBackupDirPath -Recurse -Force

  New-Item -Path $upgradeDirPath -ItemType Directory -Force | Out-Null
  (New-Object System.Net.WebClient).DownloadFile($upgradeArchiveUrl, $upgradeArchivePath)

  $7ZipPath = Resolve-Path '/signageos/7zip/7z.exe'
  . $7ZipPath x $upgradeArchivePath -o"$upgradeDirPath" -r -y

  # If bundlet applet there is change in upgrade path
  $upgradeContentPath = Join-Path $upgradeDirPath "Content"
  if (Test-Path $upgradeContentPath) {
    $upgradeDirName = "upgrade/Content"  
    $upgradeDirPathContent = (Join-Path $baseDirPath $upgradeDirName)
    $upgradeServerArchivePathContent = (Join-Path $upgradeDirPathContent $upgradeServerArchiveName)

    . $7ZipPath x $upgradeServerArchivePathContent -o"$upgradeServerDirPath" -r -y
  } else {
    . $7ZipPath x $upgradeServerArchivePath -o"$upgradeServerDirPath" -r -y
  }

  $upgradeContentPath = Join-Path $upgradeDirPath "Content"
  if (Test-Path $upgradeContentPath) {
    Get-ChildItem -Path $upgradeContentPath | Move-Item -Destination $upgradeDirPath -Force
    Remove-Item $upgradeContentPath -Force -Recurse
  }
} Catch {
  Write-ErrorToHost "Upgrade failed before new content was put in place. Server files are intact."
  Write-ErrorToHost ($ErrorFormatString -F $_.Exception.Message, $_.InvocationInfo.PositionMessage, $_.CategoryInfo.ToString(), $_.FullyQualifiedErrorId)

  Invoke-RestartComputer
}

Write-Debug "Upgrade package prepared."

Try {
  Import-Module "$PSScriptRoot/UserShell.psm1" -Force
  Clear-UserShell $kioskUsername
} Catch {
  Write-ErrorToHost "Clearing user shell failed. This will affect the process only if downgrading."
  Write-ErrorToHost ($ErrorFormatString -F $_.Exception.Message, $_.InvocationInfo.PositionMessage, $_.CategoryInfo.ToString(), $_.FullyQualifiedErrorId)
}

Try {
  & $upgradeServerDirPath\powershell\upgrade\Run-Upgrade.ps1 -upgradeDirPath $upgradeDirPath -baseDirPath $baseDirPath -upgradeServerDirPath $upgradeServerDirPath
} Catch {
  If (Test-Path $upgradeFailedDirPath) {
    Get-ChildItem -Recurse $upgradeFailedDirPath | Remove-Item -Recurse -Force -Confirm:$false
  } Else {
    New-Item -Path $baseDirPath -ItemType Directory -Name $upgradeFailedDirName -Force | Out-Null
  }

  Copy-Item -Path "$serverDirPath\*" -Destination $upgradeFailedDirPath -Recurse -Force

  Stop-ServicesSafe -Names $daemonServiceId, $daemonScreenshotServiceId

  Copy-Item -Path "$serverBackupDirPath\*" -Destination $serverDirPath -Recurse -Force

  .$daemonService uninstall
  .$daemonScreenshotService uninstall
  .$daemonService install
  .$daemonScreenshotService install

  Write-ErrorToHost "Upgrade failed after new content was put in place. Server files backup was restored."
  Write-ErrorToHost ($ErrorFormatString -F $_.Exception.Message, $_.InvocationInfo.PositionMessage, $_.CategoryInfo.ToString(), $_.FullyQualifiedErrorId)

  Invoke-RestartComputer
}
