"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFinalKey = exports.getCacheKey = exports.extractKey = void 0;
const crypto_1 = require("crypto");
const serialize = require("serialize-javascript");
/**
 * extractKey - If data should be stored in a hash, this would be the name of the hash
 *
 * @param passedInKey The desired key, or function to build the key based on arguments/context
 * @param args        The arguments the decorated method was called with
 * @param context     The instance whose method is being called
 *
 * @returns {String}
 */
const extractKey = (passedInKey = '', args, context, returnValue) => {
    // If the user passed in a cacheKey, use that. If it's a string, use it directly.
    // In the case of a function, we'll use the result of the called function.
    return passedInKey instanceof Function ? passedInKey(args, context, returnValue) : passedInKey;
};
exports.extractKey = extractKey;
/**
 * getCacheKey - Determines the cache key to use. Either from an argument extractor function,
 *               string, or number that is passed in, or a hash based on args decorated function
 *               was called with (by default)
 *
 * @param methodName  The name of the method being called
 * @param passedInKey The desired key, or function to build the key based on arguments
 * @param args        The arguments the decorated method was called with
 * @param context     The instance whose method is being called
 *
 * @returns {String}
 */
const getCacheKey = (passedInKey = '', methodName, args, context, returnValue) => {
    // If the user passed in a cacheKey, use that. If it's a string, use it directly.
    // In the case of a function, we'll use the result of the called function.
    if (passedInKey) {
        return exports.extractKey(passedInKey, args, context, returnValue);
    }
    // Fall back to a default value (md5 hash of serialized arguments and context,
    // which is the instance the method was called from)
    const callMap = {
        args,
        methodName,
        context,
    };
    const serializedKey = serialize(callMap);
    return crypto_1.createHash('md5').update(serializedKey).digest('hex');
};
exports.getCacheKey = getCacheKey;
const getFinalKey = (passedCacheKey = '', passedHashKey = '', methodName, args, context, returnValue) => {
    const cacheKey = exports.getCacheKey(passedCacheKey, methodName, args, context, returnValue);
    const hashKey = exports.extractKey(passedHashKey, args, context, returnValue);
    if (Array.isArray(cacheKey)) {
        return cacheKey.map((key) => (hashKey ? `${hashKey}:${key}` : key));
    }
    return hashKey ? `${hashKey}:${cacheKey}` : cacheKey;
};
exports.getFinalKey = getFinalKey;
//# sourceMappingURL=getCacheKey.js.map