"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const ISocket_1 = require("../ISocket");
const progressiveWait_1 = require("../../../Timer/progressiveWait");
const debug_1 = __importDefault(require("debug"));
const debug = (0, debug_1.default)('@signageos/lib:WebSocket:Client:WS:AutoReconnectingWSSocket');
const DEFAULT_INITIAL_RECONNECT_TIMEOUT = 1e3;
const DEFAULT_MAX_RECONNECT_TIMEOUT = 30e3;
class AutoReconnectingWSSocket {
    constructor(createSocket, initialReconnectTimeout = DEFAULT_INITIAL_RECONNECT_TIMEOUT, maxReconnectTimeout = DEFAULT_MAX_RECONNECT_TIMEOUT) {
        this.createSocket = createSocket;
        this.closed = false;
        this.eventListeners = [];
        this.emitsQueue = [];
        this.progressiveWait = (0, progressiveWait_1.createProgressiveWait)(initialReconnectTimeout, 2, maxReconnectTimeout);
        this.internalEventEmitter = new events_1.EventEmitter();
        this.openNewSocket();
    }
    on(event, listener) {
        this.eventListeners.push({ type: 'on', event, listener });
        this.socket.on(event, listener);
        debug('Number of listeners after on', this.eventListeners.length);
    }
    once(event, listener) {
        this.eventListeners.push({ type: 'once', event, listener });
        this.registerOnceListenerOnCurrentSocket(event, listener);
        debug('Number of listeners after once', this.eventListeners.length);
    }
    emit(event, message, callback) {
        try {
            this.socket.emit(event, message, callback);
        }
        catch (error) {
            if (error instanceof ISocket_1.UndeliveredEmitError) {
                const emitData = { event, message, callback };
                debug('Emit message failed', emitData, error);
                this.emitsQueue.push(emitData);
                debug('Number of emits in queue after fail', this.emitsQueue.length);
            }
            else {
                throw error;
            }
        }
    }
    removeListener(event, listener) {
        this.removeEventListenerFromList(event, listener);
        this.socket.removeListener(event, listener);
        debug('Number of listeners after remove', this.eventListeners.length);
    }
    removeAllListeners() {
        this.eventListeners = [];
        this.socket.removeAllListeners();
    }
    close() {
        this.closed = true;
        this.socket.close();
    }
    onConnected(callback) {
        this.internalEventEmitter.on('connected', callback);
    }
    onDisconnected(callback) {
        this.internalEventEmitter.on('disconnected', callback);
    }
    onError(callback) {
        this.internalEventEmitter.on('error', callback);
    }
    openNewSocket() {
        this.socket = this.createSocket(() => this.onSocketConnected(), () => this.onSocketDisconnected(), (error) => this.internalEventEmitter.emit('error', error));
        for (let eventListener of this.eventListeners) {
            if (eventListener.type === 'on') {
                this.socket.on(eventListener.event, eventListener.listener);
            }
            else {
                this.registerOnceListenerOnCurrentSocket(eventListener.event, eventListener.listener);
            }
        }
    }
    onSocketConnected() {
        debug('Internal socket connected');
        this.internalEventEmitter.emit('connected');
        this.progressiveWait.reset();
        // Try repeat only the current queue count from beggining (to prevent infinite loop)
        const repeatCount = this.emitsQueue.length;
        for (let i = 0; i < repeatCount; i++) {
            const emitData = this.emitsQueue.shift();
            if (emitData) {
                try {
                    this.emit(emitData.event, emitData.message, emitData.callback);
                }
                catch (error) {
                    debug('Could not repeat emit because of fatal failure', emitData, error);
                    console.error(error);
                }
            }
        }
        debug('Number of emits in queue after repeat', this.emitsQueue.length);
    }
    onSocketDisconnected() {
        return __awaiter(this, void 0, void 0, function* () {
            debug('Internal socket disconnected');
            this.internalEventEmitter.emit('disconnected');
            if (!this.closed) {
                this.socket.removeAllListeners();
                yield this.progressiveWait.wait();
                this.openNewSocket();
            }
        });
    }
    registerOnceListenerOnCurrentSocket(event, listener) {
        this.socket.once(event, listener);
        this.socket.once(event, () => this.removeEventListenerFromList(event, listener));
    }
    removeEventListenerFromList(event, listener) {
        this.eventListeners = this.eventListeners.filter((eventListener) => {
            return eventListener.event !== event || eventListener.listener !== listener;
        });
    }
}
exports.default = AutoReconnectingWSSocket;
//# sourceMappingURL=AutoReconnectingWSSocket.js.map