import { IAMQPConnection } from '../../amqpConnectionFactory';
import IEvent, { IEventPayload } from '../IEvent';
export interface LockedBindOptions {
    /**
     * Number of events to prefetch from the main queue.
     * This number has to greater than 0.
     * Make sure that you have enough main prefetch count that can properly utilize the distributed prefetch count
     * for all started consumers. The general rule is to have at least the same number of main prefetch count as the number of consumers times distributed prefetch count.
     * In other words: `mainPrefetchCount >= numberOfConsumers * distributedPrefetchCount`
     */
    mainPrefetchCount?: number;
    /**
     * Number of events to prefetch from the distributed queue.
     * This number can be 0, which means that the distributed consumption is disabled.
     * In this case, the events are consumed directly from the main queue and the distributed queue is not used (neither created).
     * If this number is greater than 0, the distributed consumption is enabled and the events are distributed from the main queue to the distributed queue.
     * You can scale the number of consumers that multiply the total number of possibly concurrently processed events.
     */
    distributedPrefetchCount?: number;
    /**
     * Delay in milliseconds before redelivering the event in case of error.
     * It's dead-lettered on the first error and redirected to the event_rejected queue for the specified delay.
     */
    redeliverDelayMs?: number;
    /**
     * If true, binding is done with notification option, so the exchange is prefixed with `events_notifications-` instead of standard `events-`.
     * @deprecated This logic is about to be removed soon. Use it temporarily only for platform-consumer-redis until it's migrated to the new logic.
     * 	The new logic is based on strong consistency awaiting.
     */
    legacyNotification?: boolean;
}
/**
 * Event consumer that consumes events in a locked manner.
 *
 * It ensures that only one event with the same lockKey is processed at a time.
 * Practically, it means that events with the same lockKey are processed sequentially.
 * However, events with different lockKeys can be processed in parallel.
 *
 * Internal implementation uses 2 separate queues (if distributedPrefetchCount > 0):
 * - Main queue for distributing events to distributed queue
 * - Distributed queue for processing events
 *
 * The main queue is "Single Active Consumer" and is locking the event processing based on lockKeys field of the event.
 * So the order of events is preserved based on the lockKeys.
 * However, the distributed queue is not locking the event processing and is processing across multiple consumers at the same time.
 * Messages in the distributed queue are transient. In case of failure, they are not lost because the main queue is persistent
 * and is not acked until the distributed queue acks the message.
 */
export declare class LockedEventConsumer {
    private amqpConnection;
    private withLock;
    constructor(amqpConnection: IAMQPConnection);
    bind<TPayload extends IEventPayload>(eventTypes: string[], domainName: string, consumerType: string, onEvent: (event: IEvent<TPayload>) => Promise<void>, bindOptions?: LockedBindOptions): Promise<() => Promise<() => Promise<void>>>;
    private createMainBindOptions;
    private createConsumeEvent;
    private createDistributedBindOptions;
    private createDistributeEvent;
}
