/// <reference types="lib.es2017.observable" />
import { SubscriptionType as SubscriptionTypeInternal } from '@signageos/common-types/dist/Subscription/subscription';
import { BillingPlan } from '@signageos/common-types/dist/Billing/Billing';
import { OrganizationReporters } from '@signageos/common-types/dist/Reporters/organizationReportersTypes';
import { Role } from '@signageos/common-types/dist/Account/Account';
import IChanges from '../../IChanges';
import { IAccount } from '../Account/accountModel';
import { ICompanyNetwork } from '../CompanyNetwork/companyNetworkModel';
import { IWhiteLabelSettings } from '../WhiteLabelSettings/whiteLabelSettingsModel';
import { IPolicy } from '../Policy/policyModel';
import { IDevice } from '../Device/deviceModel';
import { OffsetPagination } from '../../Lib/Pagination/paginationTypes';
import { LocationMarker } from '@signageos/common-types/dist/Location/location';
/** @deprecated Use import from '@signageos/common-types/dist/Subscription/subscription' instead */
export declare type SubscriptionType = SubscriptionTypeInternal;
/** @deprecated Use import from '@signageos/common-types/dist/Subscription/subscription' instead */
export declare const SubscriptionType: typeof SubscriptionTypeInternal;
export declare enum OrderDirection {
    ASC = 1,
    DESC = -1
}
export interface IOrganizationPropertyFilter {
    uid?: string;
    parentUid?: string;
    name?: string;
    accountId?: number;
    isCompany?: boolean;
    billingPlan?: BillingPlan;
    companyNetworkUids?: ICompany['companyNetworkUids'];
    trialPeriodEndedAtRange?: {
        from: IOrganization['trialPeriodEndedAt'];
        to: IOrganization['trialPeriodEndedAt'];
    };
}
declare type OwnFieldSorterType = keyof Pick<IOrganization, 'title' | 'createdAt' | 'uid' | 'subscriptionType'>;
declare type FieldSorterType = OwnFieldSorterType | 'company';
export interface IOrganizationListSorter {
    field: FieldSorterType;
    direction: 'ascending' | 'descending';
}
export interface IOrganizationListPagination {
    offset?: number;
    limit?: number;
}
export interface IOrganizationFetchOptions {
    /** Fetch organizations that belong to a specific account */
    account: IAccount;
    filter?: IOrganizationPropertyFilter;
    sorter?: IOrganizationListSorter;
    pagination?: IOrganizationListPagination;
}
export interface IAllCompaniesFetchOptions {
    search?: string;
    filter?: IOrganizationPropertyFilter;
    sorter?: IOrganizationListSorter;
    pagination?: IOrganizationListPagination;
}
export interface IBillingInfo {
    companyName: string;
    taxId?: string;
    street: string;
    street2?: string;
    postcode: string;
    city: string;
    state: string;
    country: string;
    email?: string;
    phone?: string;
    contactPersonName?: string;
    orderID?: string;
}
export declare type EmailDomain = string;
export interface ICompany {
    billingPlan?: BillingPlan;
    billingInfo?: IBillingInfo;
    /** indicate SignageOs company */
    isSuper?: boolean;
    deviceNameTemplate?: string;
    companyNetworkUids?: string[];
    whiteLabelSettingsUid?: IWhiteLabelSettings['uid'];
    emailDomains?: EmailDomain[];
    defaultAccountRole?: Role | null;
    iconUrl?: string | null;
}
/**
 * IOrganization stands for both companies and organizations entities
 * Some fields necessity may vary depending whether the real entity is company or organization
 *
 * Organization has always parentUid of parent company and the relation can go only one level deep
 */
export interface IOrganization extends ICompany {
    uid: string;
    /**
     * Subdomain prefix
     * Must contain only alphanumeric and characters dived by dashes
     * e.g.: broadsign or mc-donald
     * */
    name: string;
    title: string;
    createdAt: Date;
    oauthClientId: string;
    oauthClientSecret: string;
    isCompany: boolean;
    parentUid?: string;
    subscriptionType?: SubscriptionType /** relevant for organizations only */;
    defaultSubscriptionType?: SubscriptionType /** relevant for organizations only */;
    accountIds?: number[] /** used for reverse organization-account mapping */;
    productionSince?: Date;
    maxDevicesCount?: number;
    trialPeriodEndedAt?: Date;
    reporters?: OrganizationReporters;
    vpnServerNameTag?: string;
    defaultLocationMarker?: LocationMarker | null;
}
export declare type IOrganizationIdentification = Pick<IOrganization, 'uid'>;
/**
 * interface used in lookup query to fetch all organizations with its policies and devices
 */
export interface IOrganizationPolicyDevice extends IOrganization {
    orgPolicies: (IPolicy & {
        devices: IDevice[];
    })[];
}
/**
 * organization param is company
 */
export interface IUpdateCompanyParams<TOrganization> {
    organization: TOrganization;
    title?: IOrganization['title'];
    emailDomains?: ICompany['emailDomains'];
    defaultAccountRole?: ICompany['defaultAccountRole'];
    iconUrl?: string | null;
}
export interface IAssignCompanyNetworkToOrganizationParams<TOrganization> {
    organization: TOrganization;
    companyNetwork: ICompanyNetwork;
}
export interface IUnassignCompanyNetworkFromOrganizationParams<TOrganization> {
    organization: TOrganization;
    companyNetwork: ICompanyNetwork;
}
export interface IAssignWhiteLabelSettingsToOrganizationParams<TOrganization> {
    organization: TOrganization;
    whiteLabelSettings: IWhiteLabelSettings;
}
export interface FetchCompaniesByEmailDomainsParams {
    emailDomains: EmailDomain[];
}
export interface FetchCompanyByEmailDomainPrams {
    emailDomain: EmailDomain;
}
export interface IDistributionByBillingPlan {
    [BillingPlan.Open]: number;
    [BillingPlan.Starter]: number;
    [BillingPlan.Medium]: number;
    [BillingPlan.Enterprise]: number;
}
export declare function createDefaultDistributionByBillingPlan(): IDistributionByBillingPlan;
export interface IDistributionByDevicePlan {
    [SubscriptionType.Open]: number;
    [SubscriptionType.Basic]: number;
    [SubscriptionType.Medium]: number;
    [SubscriptionType.All]: number;
    [SubscriptionType.Platform]?: number;
}
export declare function createDefaultDistributionByDevicePlan(): IDistributionByDevicePlan;
export declare type IOrganizationModel<TOrganization extends IOrganization> = IOrganizationModelRequired<TOrganization> & IOrganizationModelOptional<TOrganization> & IOrganizationModelObservable<TOrganization> & IOrganizationModelStatistics<TOrganization> & IOrganizationModelTree<IOrganization>;
export interface IOrganizationModelRequired<TOrganization extends IOrganization> extends IOrganizationModelWrite<TOrganization> {
    fetchByUid(uid: string): Promise<TOrganization | null>;
    fetchByOAuthClientId(oauthClientId: string): Promise<TOrganization | null>;
    fetchByName(name: IOrganization['name']): Promise<TOrganization | null>;
    fetchListByUids(uids: string[]): Promise<TOrganization[]>;
    fetchAllList(): Promise<TOrganization[]>;
    fetchAllOrganizationsList(): Promise<TOrganization[]>;
    fetchAllCompaniesList(): Promise<TOrganization[]>;
    fetchAllByFilter(options: IAllCompaniesFetchOptions): Promise<TOrganization[]>;
    countAllByFilter(options: IAllCompaniesFetchOptions): Promise<number>;
    fetchSortedListByFilterAndAccount(options: IOrganizationFetchOptions): Promise<{
        companies: TOrganization[];
        organizations: TOrganization[];
        organizationsTotalCount: number;
    }>;
    fetchTotalCountByAccount(account: IAccount, propertyFilter?: IOrganizationPropertyFilter): Promise<number>;
    fetchOrganizationCompanyMap(uids: string[]): Promise<Map<string, string>>;
    getIterator(): Promise<AsyncIterableIterator<IOrganization>>;
    getAllOrganizationsWithPoliciesAndDevicesIterator(): Promise<AsyncIterableIterator<IOrganizationPolicyDevice>>;
    fetchCompaniesByEmailDomains(params: FetchCompaniesByEmailDomainsParams): Promise<TOrganization[]>;
    fetchCompanyByEmailDomain(params: FetchCompanyByEmailDomainPrams): Promise<TOrganization | null>;
}
export interface IOrganizationModelWrite<TOrganization extends IOrganization> {
    createCompany(uid: string, name: string, title: string, oauthClientId: string, oauthClientSecret: string, createdAt: Date, billingPlan: BillingPlan, iconUrl?: string): Promise<void>;
    /**
     * @params organization param is company
     */
    updateCompany(params: IUpdateCompanyParams<TOrganization>): Promise<void>;
    createOrganization(uid: string, name: string, title: string, oauthClientId: string, oauthClientSecret: string, createdAt: Date, subscriptionType: SubscriptionType, iconUrl?: string): Promise<void>;
    updateTitle(organizationRow: IOrganizationIdentification, title: string): Promise<void>;
    updateIconUrl(organizationRow: IOrganizationIdentification, iconUrl: string | null): Promise<void>;
    updateDefaultLocationMarker(organizationRow: IOrganizationIdentification, defaultLocationMarker: LocationMarker | null): Promise<void>;
    updateVPNUrl(organizationRow: IOrganizationIdentification, vpnServerNameTag: string): Promise<void>;
    deleteVPNUrl(organizationRow: IOrganizationIdentification): Promise<void>;
    /** relevant for organizations only */
    changeSubscriptionType(organizationRow: IOrganizationIdentification, subscriptionType: SubscriptionType): Promise<void>;
    /** relevant for companies only */
    updateBillingInfo(organizationRow: IOrganizationIdentification, billingInfo: IBillingInfo): Promise<void>;
    /** relevant for companies only */
    updateBillingPlan(organizationRow: IOrganizationIdentification, plan: string): Promise<void>;
    /** relevant for companies only */
    setDefaultSubscriptionType(organizationRow: IOrganizationIdentification, defaultSubscriptionType: SubscriptionType): Promise<void>;
    remove(organizationRow: IOrganizationIdentification): Promise<void>;
    assignAccount(organizationRow: IOrganizationIdentification, account: IAccount, assignedAt: Date): Promise<void>;
    removeAccount(organizationRow: IOrganizationIdentification, account: IAccount): Promise<void>;
    updateProductionSince(organizationRow: IOrganizationIdentification, productionSince: Date): Promise<void>;
    deleteProductionSince(organizationRow: IOrganizationIdentification): Promise<void>;
    updateMaxDevicesCount(organizationRow: IOrganizationIdentification, maxDevicesCount: number): Promise<void>;
    updateEndOfTrialPeriod(organizationRow: IOrganizationIdentification, endDate: Date): Promise<void>;
    clearEndOfTrialPeriod(organizationRow: IOrganizationIdentification): Promise<void>;
    setDeviceNameTemplate(organizationRow: IOrganizationIdentification, template: string): Promise<void>;
    assignCompanyNetworkToOrganization({ organization, companyNetwork, }: IAssignCompanyNetworkToOrganizationParams<IOrganizationIdentification>): Promise<void>;
    unassignCompanyNetworkFromOrganization({ organization, companyNetwork, }: IUnassignCompanyNetworkFromOrganizationParams<IOrganizationIdentification>): Promise<void>;
    assignWhiteLabelSettingsToOrganization({ organization, whiteLabelSettings, }: IAssignWhiteLabelSettingsToOrganizationParams<IOrganizationIdentification>): Promise<void>;
    unassignWhiteLabelSettingsFromOrganization(organization: IOrganizationIdentification): Promise<void>;
    updateOrganizationReporters(organizationRow: IOrganizationIdentification, reporters: OrganizationReporters): Promise<void>;
}
export interface IOrganizationModelOptional<TOrganization extends IOrganization> {
    fetchListByAccount(account: IAccount): Promise<TOrganization[]>;
    fetchListByAccountAndFilter(account: IAccount, propertyFilter: IOrganizationPropertyFilter): Promise<TOrganization[]>;
    fetchListByPropertyFilter(propertyFilter: IOrganizationPropertyFilter): Promise<TOrganization[]>;
}
export interface IOrganizationModelStatistics<TOrganization extends IOrganization> {
    fetchCompaniesDistributionByBillingPlan(createdSince?: Date, createdBefore?: Date): Promise<IDistributionByBillingPlan>;
    fetchOrganizationsDistributionByDevicePlan(isProduction?: boolean, createdSince?: Date, createdBefore?: Date): Promise<IDistributionByDevicePlan>;
    fetchOrganizationByUidDevicesCount(organizationUid: string): Promise<(TOrganization & {
        deviceCount: number;
    }) | undefined>;
    fetchOrganizationsByCompanyUid(companyUid: string): Promise<TOrganization[]>;
}
export interface IOrganizationModelTree<TOrganization extends IOrganization> {
    isParent(organizationRow: TOrganization): boolean;
    isChild(organizationRow: TOrganization): boolean;
    getParent(organizationRow: TOrganization): Promise<TOrganization>;
    getChildren(organizationRow: TOrganization, pagination?: OffsetPagination, sorter?: IOrganizationListSorter): Promise<TOrganization[]>;
    getChildrenCount(organizationRow: TOrganization): Promise<number>;
    setParent(organizationRow: IOrganizationIdentification, parentalOrganizationRow: IOrganizationIdentification): Promise<void>;
    unsetParent(organizationRow: IOrganizationIdentification): Promise<void>;
    getChildrenWithFilter(organizationRow: TOrganization, filter: IOrganizationPropertyFilter): Promise<TOrganization[]>;
}
export interface IOrganizationModelObservable<T extends IOrganization> {
    waitByName(name: string): Promise<T>;
    observeByUid(uid: string): Promise<Observable<IChanges<T>, Error>>;
    observeAllOrganizationsList(): Promise<Observable<IChanges<T>, Error>>;
    observeListByAccount(options: IOrganizationFetchOptions): Promise<Observable<{
        organizations: T[];
        companies: T[];
        organizationsTotalCount: number;
    }, Error>>;
}
export declare function validateSubscriptionType(subscriptionType: SubscriptionType): void;
export declare type OrganizationsDeviceCountsStatisticsType = {
    [organizationUid: string]: IDistributionByDevicePlan;
};
export {};
//# sourceMappingURL=organizationModel.d.ts.map